package csdk.v1_0.runner.application;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.nio.charset.Charset;
import java.util.Map;

import tecgraf.javautils.core.lng.LNG;
import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IApplication;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.core.IContext;
import csdk.v1_0.api.filesystem.project.IProjectContext;
import csdk.v1_0.api.filesystem.project.IProjectObserver;
import csdk.v1_0.runner.ApplicationManager;
import csdk.v1_0.runner.ApplicationRegistry;
import csdk.v1_0.runner.IContextFactory;
import csdk.v1_0.runner.Runner;
import csdk.v1_0.runner.core.RunnerEnvironment;

/**
 * Classe que mapeia uma aplicao baseada no CSDK para a sandbox.
 * 
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class RunnerApplication implements IApplication {

  /**
   * Prefixo para chave de internacionalizao das mensagens de erro.
   */
  private final static String ERROR_PREFIX = RunnerApplication.class
    .getSimpleName();

  /**
   * Contador usado para gerar os identificadores das instncias.
   */
  private static int instanceCounter = 0;

  /**
   * A aplicao propriamente dita.
   */
  private final IApplication application;

  /**
   * A interface com o ambiente CSDK do {@link Runner}.
   */
  private final RunnerEnvironment runnerEnv;

  /**
   * Construtor.
   * 
   * @param registry o registro da aplicao.
   * @param charset charset do ambiente.
   * @throws ApplicationException em caso de erro ao instanciar a aplicao.
   */
  public RunnerApplication(ApplicationRegistry registry, Charset charset)
    throws ApplicationException {
    String id = registry.getApplicationId();
    String instanceId = generateInstanceId(id);
    this.runnerEnv = createCSDKEnvironment(instanceId, registry, charset);
    Class<? extends IApplication> appClass = loadApplicationClass(registry);
    if (appClass == null) {
      throw new ApplicationException("No class found for " + id + "!");
    }
    try {
      Constructor<?> constructor =
        appClass.getConstructor(ICSDKEnvironment.class);
      this.application = (IApplication) constructor.newInstance(this.runnerEnv);
    }
    catch (final InvocationTargetException e) {
      final Throwable appException = e.getTargetException();
      final String message =
        LNG.get(ERROR_PREFIX + ".instatiation.exception", new Object[] { id,
            appClass.getName(), appException.getLocalizedMessage() });
      throw new ApplicationException(message, appException);
    }
    catch (final NoSuchMethodException e) {
      final String message =
        LNG.get(ERROR_PREFIX + ".no.valid.constructor", new Object[] { id,
            appClass.getName() });
      throw new ApplicationException(message, e);
    }
    catch (final InstantiationException e) {
      final String message =
        LNG.get(ERROR_PREFIX + ".no.instantiation", new Object[] { id,
            appClass.getName() });
      throw new ApplicationException(message, e);
    }
    catch (final IllegalAccessException e) {
      final String message =
        LNG.get(ERROR_PREFIX + ".application.manager.bad.implementation",
          new Object[] { id, appClass.getName() });
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Faz a carga da classe (com base no nome). Esta funo utiliza
   * <b>extraordinariamente</b> a anotao de supresso de avisos (warnings)
   * porque foi necessrio fazer um <i>loadClass</i>.
   * 
   * (Ver definio de {@link ClassLoader#loadClass(String)})
   * 
   * @param reg registro da aplicao.
   * 
   * @return a classe.
   * @throws ApplicationException se a classe da aplicao no for encontrada ou
   *         vlida.
   */
  @SuppressWarnings("unchecked")
  private Class<? extends IApplication> loadApplicationClass(
    ApplicationRegistry reg) throws ApplicationException {
    try {
      ClassLoader classloader = reg.getClassloader();
      final Class<?> loadedClass =
        Class.forName(reg.getClassName(), true, classloader);
      if (IApplication.class.isAssignableFrom(loadedClass)) {
        return (Class<? extends IApplication>) loadedClass;
      }
      else {
        throw new IllegalArgumentException(
          "Classe principal da aplicao no implementa a interface "
            + IApplication.class.getName());
      }
    }
    catch (Exception e) {
      throw new ApplicationException(
        "Erro ao carregar a classe principal da aplicao", e);
    }
  }

  /**
   * Cria o ambiente de interface com o CSDK para a aplicao.
   * 
   * @param instanceId identificador da instncia.
   * @param registry o registro da aplicao.
   * @param charset charset do ambiente.
   * @return o ambiente.
   * 
   * @throws ApplicationException em caso de erro ao criar o ambiente.
   */
  private RunnerEnvironment createCSDKEnvironment(String instanceId,
    ApplicationRegistry registry, Charset charset) throws ApplicationException {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    IContextFactory contextFactory = appManager.getContextFactory();
    Map<Class<? extends IContext>, IContext> contexts =
      contextFactory.createCSDKContexts(instanceId, registry, appManager
        .getRunnerProperties());
    if (contexts == null || contexts.isEmpty()
      || !contexts.containsKey(IApplicationContext.class)) {
      throw new ApplicationException(
        "Context factory must create the mandatory context "
          + IApplicationContext.class.getSimpleName());
    }
    if (registry.requiresProject()) {
      if (!contexts.containsKey(IProjectContext.class)) {
        throw new ApplicationException(
          "Application cannot be run without a project context ("
            + IProjectContext.class.getSimpleName() + ")");
      }
      else {
        IProjectContext projectContext =
          (IProjectContext) contexts.get(IProjectContext.class);
        projectContext.addProjectObserver(new IProjectObserver() {
          @Override
          public void onProjectOpen(String projectId) {
            //ignora evento
          }

          @Override
          public void onProjectClose(String projectId) {
            finishApplication();
          }
        });
      }
    }
    return new RunnerEnvironment(contexts, charset.name());
  }

  /**
   * Gera o identificador nico da instncia da aplicao.
   * 
   * @param id o identificador da aplicao.
   * @return o identificador da instncia.
   */
  private synchronized static String generateInstanceId(String id) {
    return id + "_" + ++instanceCounter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    return application.canEndApplication();
  }

  /**
   * Tenta terminar a execuo da aplicao
   * 
   * @return verdadeiro se a execuo foi terminada.
   */
  public boolean finishApplication() {
    if (!canEndApplication()) {
      return false;
    }
    killApplication();
    return true;
  }

  /**
   * Termina a execuo aplicao.
   */
  private void killApplication() {
    try {
      application.onApplicationEnd();
    }
    catch (ApplicationException e) {
      String errorMsg = LNG.get(ERROR_PREFIX + ".finish.error");
      System.err.println(errorMsg);
      System.err.println(e.getMessage());
      e.printStackTrace();
    }
    finally {
      runnerEnv.cleanupContexts();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessageReceived(IMessage message, IMessageSender sender) {
    this.application.onMessageReceived(message, sender);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    this.killApplication();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    this.application.onApplicationStart();
  }

  /**
   * Obtm o id da instncia da aplicao.
   * 
   * @return o id com uma string.
   */
  public String getInstanceId() {
    IApplicationContext context =
      runnerEnv.getContext(IApplicationContext.class);
    return context.getInstanceId();
  }

  /**
   * Obtm o id da aplicao.
   * 
   * @return o id com uma string.
   */
  public String getApplicationId() {
    IApplicationContext context =
      runnerEnv.getContext(IApplicationContext.class);
    return context.getApplicationId();
  }

  /**
   * Obtm a aplicao propriamente dita.
   * 
   * @return a aplicao.
   */
  public IApplication getApplication() {
    return application;
  }

}
