package snakegame.map;

import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import javax.swing.table.DefaultTableModel;

import snakegame.algorithm.IMoveAlgorithm;
import snakegame.exception.GameOverException;
import snakegame.exception.VictoryException;
import snakegame.session.MapSize;

/**
 * Modelo da tabela que representa o mapa do jogo.
 * 
 * NOTA: Esse modelo cria um mapa quadrado.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class SnakeTableModel extends DefaultTableModel {

  /** Tamanho do mapa. */
  private final Cell[][] map;

  /** Posies que a cobra ocupa: [0] = cabea; [size-1] = cauda. */
  private List<Point> snake;

  /** Posio da comida. */
  private Point food;

  /** Direo. */
  private Direction direction;

  /** Algoritmo de movimentao. */
  private final IMoveAlgorithm algorithm;

  /**
   * Construtor.
   * 
   * @param mapSize tamanho do mapa.
   * @param fill flag que indica se  para preencher o mapa com os elementos do
   *        jogo.
   * @param algorithm algoritmo de movimentao.
   */
  public SnakeTableModel(MapSize mapSize, boolean fill, IMoveAlgorithm algorithm) {
    super(mapSize.size, mapSize.size);

    this.algorithm = algorithm;

    // cria mapa vazio.
    this.map = new Cell[getRowCount()][getColumnCount()];
    for (int i = 0; i < map.length; i++) {
      for (int j = 0; j < map[i].length; j++) {
        map[i][j] = Cell.EMPTY;
      }
    }

    if (fill) {
      int middle = getRowCount() / 2;
      snake = new ArrayList<>();
      snake.add(new Point(middle, middle));
      snake.add(new Point(middle - 1, middle));
      snake.add(new Point(middle - 2, middle));

      for (Point point : snake) {
        map[point.x][point.y] = Cell.SNAKE;
      }

      direction = Direction.DOWN;

      food = findEmptySpace();
      map[food.x][food.y] = Cell.FOOD;
    }
  }

  /**
   * Construtor.
   * 
   * @param size tamanho do mapa.
   */
  public SnakeTableModel(MapSize size) {
    this(size, false, null);
  }

  /**
   * Define a direo da cobra.
   * 
   * @param direction direo.
   */
  public void setDirection(Direction direction) {
    if (direction == null) {
      return;
    }
    else if (this.direction == Direction.DOWN && direction == Direction.UP) {
      return;
    }
    else if (this.direction == Direction.UP && direction == Direction.DOWN) {
      return;
    }
    else if (this.direction == Direction.LEFT && direction == Direction.RIGHT) {
      return;
    }
    else if (this.direction == Direction.RIGHT && direction == Direction.LEFT) {
      return;
    }
    this.direction = direction;
  }

  /**
   * Faz o movimento da cobra.
   * 
   * @throws GameOverException em caso de suicdio ou bater na parede.
   * @throws VictoryException em caso de zerar o jogo.
   */
  public void move() throws GameOverException, VictoryException {
    executeAlgorithm();

    Point nextPoint = nextPoint();
    Cell nextCell = map[nextPoint.x][nextPoint.y];

    switch (nextCell) {
      case SNAKE:
        throw new GameOverException("Suicdio.");
      case EMPTY:
        snake.add(0, nextPoint);
        map[nextPoint.x][nextPoint.y] = Cell.SNAKE;

        Point point = snake.remove(snake.size() - 1);
        map[point.x][point.y] = Cell.EMPTY;
        break;
      case FOOD:
        snake.add(0, nextPoint);
        map[nextPoint.x][nextPoint.y] = Cell.SNAKE;

        if (snake.size() == map.length * map.length) {
          throw new VictoryException("Parabns!");
        }
        food = findEmptySpace();
        map[food.x][food.y] = Cell.FOOD;
        break;
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean isCellEditable(int row, int column) {
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public Object getValueAt(int row, int column) {
    return map[row][column];
  }

  /**
   * Retorna o prximo ponto a ser visitado pela cobra.
   * 
   * @return prximo ponto.
   * 
   * @throws GameOverException caso exceda o limite do mapa.
   */
  private Point nextPoint() throws GameOverException {
    int x = snake.get(0).x;
    int y = snake.get(0).y;
    switch (direction) {
      case UP:
        x--;
        break;
      case DOWN:
        x++;
        break;
      case LEFT:
        y--;
        break;
      case RIGHT:
        y++;
        break;
    }

    if (x < 0 || x == map.length || y < 0 || y == map.length) {
      throw new GameOverException("Limite do mapa excedido.");
    }
    return new Point(x, y);
  }

  /**
   * Rotinha que acha uma clula vazia.
   * 
   * @return ponto.
   */
  private Point findEmptySpace() {
    Random random = new Random(System.currentTimeMillis());
    int x;
    int y;
    do {
      x = random.nextInt(map.length);
      y = random.nextInt(map.length);

    } while (map[x][y] != Cell.EMPTY);
    return new Point(x, y);
  }

  /** Executa o algoritmo de movimentao se este for definido. */
  private void executeAlgorithm() {
    if (algorithm == null) {
      return;

    }
    List<Point> points = Collections.unmodifiableList(snake);
    int rows = getRowCount();
    int columns = getColumnCount();

    setDirection(algorithm.whereShouldIGo(points, direction, food, rows,
      columns));
  }
}
