package tecgraf.javautils.gui;

import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

import javax.swing.SwingUtilities;

import sun.awt.AppContext;

/**
 * Classe que processa as chamadas que precisam ser executadas na thread (EDT)
 * do Swing e contorna o erro introduzido pela Oracle no update25 do Java7.
 * 
 * O erro est registrado em:
 * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=8019274
 * 
 * O cenrio onde ocorre o erro  o seguinte:
 * <ul>
 * <li>Mquina do cliente usando a JRE com Java7 update25 ou update40;
 * <li>Aplicaes iniciadas com o JavaWebStart;
 * <li>Chamadas a mtodos SwingUtilities.invokeLater, isEventDispatchThread e
 * invokeAndWait pela thread RMI criada automaticamente nas chamadas do servidor
 * para o cliente;
 * <li>Chamadas ao Swing fora da thread da EDT, em casos onde o invokeLater
 * deveria ser usado.
 * </ul>
 * 
 * 
 * A soluo utiliza uma thread auxiliar para delegar as chamadas para o
 * SwingUtilities e, portanto, permite que as chamadas atravs de uma thread RMI
 * no lanem a exceo NPE no AppContext nas aplicaes iniciadas com o
 * JavaWebStart.
 * 
 * As chamadas aos mtodos SwingUtilities.invokeLater,
 * SwingUtilities.isEventDispatchThread e SwingUtilities.invokeAndWait podem ser
 * substitudas por chamadas aos mtodos correspondentes nessa classe, que
 * possuem a mesma API.
 * 
 * O uso da soluo de contorno requer que a aplicao faa um init da classe
 * SwingThreadDispatcher. Caso no seja inicializada, o comportamento da
 * aplicao que usa essa classe utilitria  igual ao comportamento corrente,
 * ou seja, as chamadas so delegadas diretamente para os mtodos da
 * SwingUtilities e, portanto, lanam a exceo nos cenrios de ocorrncia do
 * bug.
 * 
 * 
 * @author Tecgraf PUC-Rio
 */
public class SwingThreadDispatcher {

  /**
   * Para os testes em modo desenvolvimento exibir na console java as mensagens
   * de debug quando a execuo  normal.
   */
  private static boolean DEBUG_NORMAL = false;

  /**
   * Para os testes em modo desenvolvimento exibir na console java as mensagens
   * de debug quando a execuo usa a soluo de contorno para o BUG.
   */
  private static boolean DEBUG_BUG = false;

  /**
   * Array com streams de sada usados no DEBUG
   */
  private static PrintStream[] printStreamList = null;

  /**
   * Faz a chamada ao SwingUtilities.invokeLater.
   */
  private static class InvokeLaterCallable implements Callable<Void> {
    /**
     * Objeto que implementa o Runnable original.
     */
    private Runnable doRun;

    /**
     * Construtor.
     * 
     * @param doRun implementa o Runnable original.
     */
    public InvokeLaterCallable(Runnable doRun) {
      this.doRun = doRun;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Void call() throws Exception {
      show("InvokeLaterCallable.call", "Chamando o SwingUtilities.invokeLater");
      SwingUtilities.invokeLater(doRun);
      show("InvokeLaterCallable.call",
        "SwingUtilities.invokeLater executado com sucesso");
      return null;
    }

  }

  /**
   * Faz a chamada ao SwingUtilities.invokeAndWait.
   * 
   */
  private static class InvokeAndWaitCallable implements Callable<Void> {
    /**
     * Objeto que implementa o Runnable original.
     */
    private Runnable doRun;

    /**
     * Construtor.
     * 
     * @param doRun implementa o Runnable original.
     */
    public InvokeAndWaitCallable(Runnable doRun) {
      this.doRun = doRun;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Void call() throws Exception {
      show("InvokeAndWaitCallable.call",
        "Chamando o SwingUtilities.invokeAndWait");
      SwingUtilities.invokeAndWait(doRun);
      show("InvokeAndWaitCallable.call",
        "SwingUtilities.invokeAndWait executado com sucesso");
      return null;
    }

  }

  /**
   * Servio para execuo das tarefas fora da thread que est com o AppContext
   * null.
   */
  private static ExecutorService executor;

  /**
   * Delega a execuo de um objeto Runnable para a EDT do Swing.
   * 
   * Se o AppContext no for null, o resultado  delegado para o comportamento
   * default de SwingUtilities.invokeLater.
   * 
   * Se o AppContext for null, delega para uma tarefa futura que  executada
   * pela thread do Executor criado no init. SwingUtilities.invokeLater.
   * 
   * @param doRun o objeto Runnable que  despachado para execuo na EDT do
   *        Swing.
   * 
   */
  public static void invokeLater(Runnable doRun) {
    if (AppContext.getAppContext() != null) {
      showDefault("SwingThreadDispatcher.invokeLater",
        "AppContext nao e' null portanto delega para "
          + "SwingUtilities.invokeLater");
      SwingUtilities.invokeLater(doRun);
      return;
    }
    show("SwingThreadDispatcher.invokeLater",
      "AppContext e' null portanto usa uma FutureTask para executar");
    FutureTask<Void> task = new FutureTask<Void>(new InvokeLaterCallable(
      doRun));
    executor.submit(task);
  }

  /**
   * Se o AppContext for null, podemos ter certeza que no estamos na EDT e,
   * portanto, retorna sempre false.
   * 
   * Se o AppContext no for null, o resultado  delegado para o comportamento
   * default de SwingUtilities.isEventDispatchThread
   * 
   * @return {@code true} se a execuo est na thread EDT e {@code false} caso
   *         contrrio
   */
  public static boolean isEventDispatchThread() {
    if (AppContext.getAppContext() != null) {
      boolean result = SwingUtilities.isEventDispatchThread();
      showDefault("SwingThreadDispatcher.isEventDispatchThread",
        "AppContext nao e' null portanto delega para "
          + "SwingUtilities.isEventDispatchThread: " + result);
      return result;

    }
    show("SwingThreadDispatcher.isEventDispatchThread",
      "AppContext e' null portanto retorna false.");
    return false;
  }

  /**
   * Delega a execuo de um objeto Runnable para a EDT do Swing usando o
   * SwingUtilities.invokeAndWait.
   * 
   * Se o AppContext no for null, o resultado  delegado para o comportamento
   * default de SwingUtilities.invokeAndWait.
   * 
   * Se o AppContext for null, usa uma FutureTask para submeter a um servio de
   * execuo.
   * 
   * @param doRun o objeto Runnable que  despachado para execuo na EDT do
   *        Swing.
   * @throws InterruptedException a mesma exceo InterruptedException lanada
   *         no SwingUtilities.invokeAndWait
   * @throws InvocationTargetException a mesma exceo InterruptedException
   *         lanada no SwingUtilities.invokeAndWait
   * 
   */
  public static void invokeAndWait(Runnable doRun) throws InterruptedException,
    InvocationTargetException {
    if (AppContext.getAppContext() != null) {
      showDefault("SwingThreadDispatcher.invokeAndWait",
        "AppContext nao e' null portanto delega para "
          + "SwingUtilities.invokeAndWait.");
      SwingUtilities.invokeAndWait(doRun);
      return;
    }
    show("SwingThreadDispatcher.invokeAndWait",
      "AppContext e' null portanto usa uma FutureTask para aguardar");
    FutureTask<Void> task = new FutureTask<Void>(new InvokeAndWaitCallable(
      doRun));
    executor.submit(task);
    try {
      task.get();
    }
    catch (ExecutionException e) {
      Throwable cause = e.getCause();
      if (cause.getClass().equals(InvocationTargetException.class)) {
        show("SwingThreadDispatcher.invokeAndWait",
          "Exceo InvocationTargetException.");
        throw (InvocationTargetException.class.cast(cause));
      }
      if (cause.getClass().equals(InterruptedException.class)) {
        show("SwingThreadDispatcher.invokeAndWait",
          "Exceo InterruptedException.");
        throw (InterruptedException.class.cast(cause));
      }
      throw new RuntimeException(
        "Ocorreu um erro na tentativa de usar o executor.", e);
    }
  }

  /**
   * Inicializao do executor para processamento das chamadas ao EDT. O init
   * deve ser executado antes que o cliente interaja com o servidor por meio de
   * observadores a serem gatilhados por RMI (contexto do bug).
   * 
   * No pode ser chamado quando o AppContext for null porque seno, o thread
   * group do executor criado tambm mantm o AppContext null e, portanto, as
   * chamadas ao mtodo SwingUtilities.invokeLater lanar a exceo NPE.
   */
  public static void init() {
    if (AppContext.getAppContext() == null) {
      show("SwingThreadDispatcher.init",
        "AppContext e' null e o init nao pode ser chamado.");
      throw new RuntimeException(
        "O init nao pode ser chamado com appContext null");
    }
    if (executor != null) {
      show("SwingThreadDispatcher.init",
        "AppContext nao e' null mas o init ja' foi chamado.");
      return;
    }
    show("SwingThreadDispatcher.init",
      "AppContext nao e' null. Cria e inicia o executor de processamento.");
    executor = Executors.newSingleThreadExecutor();
    Future<Boolean> future = executor.submit(new Callable<Boolean>() {
      @Override
      public Boolean call() throws Exception {
        return AppContext.getAppContext() == null;
      }
    });
    try {
      if (future.get()) {
        throw new RuntimeException("O executor est com appContext null.");
      }
      show("SwingThreadDispatcher.init",
        "O executor foi criado corretamente com AppContext nao null.");
    }
    catch (Exception e) {
      throw new RuntimeException(
        "Ocorreu um erro na tentativa de usar o executor.", e);
    }
  }

  /**
   * Finaliza as atividades de interfacee
   */
  public static void shutdown() {
    executor.shutdown();
  }

  /**
   * Mostra uma mensagem na console se o debug estiver habilitado.
   * 
   * @param methodName nome do mtodo
   * @param msg mensagem a ser exibida na console
   */
  private static void show(String methodName, String msg) {
    if (DEBUG_BUG) {
      if (printStreamList == null) {
        printStreamList = new PrintStream[] { System.out };
      }
      for (PrintStream o : printStreamList) {
        o.println(">> " + methodName + ": " + msg);
      }
    }
  }

  /**
   * Mostra uma mensagem na console se o debug estiver habilitado.
   * 
   * @param methodName nome do mtodo
   * @param msg mensagem a ser exibida na console
   */
  private static void showDefault(String methodName, String msg) {
    if (DEBUG_NORMAL) {
      if (printStreamList == null) {
        printStreamList = new PrintStream[] { System.out };
      }
      for (PrintStream o : printStreamList) {
        o.println(methodName + ": " + msg);
      }
    }
  }

  /**
   * Atribui uma lista de stream para saida do debug.
   * 
   * @param outputStreamList o array com a lista de streams usados no debug
   */
  public static void setPrinter(PrintStream... outputStreamList) {
    printStreamList = outputStreamList;
  }

  /**
   * Configura os modos de debug para exibir mensagens na console durante os
   * testes em desenvolvimento. O default  estar desabilitado.
   * 
   * @param error para os testes em modo desenvolvimento exibir na console java
   *        as mensagens de debug quando a execuo usa a soluo de contorno
   *        para o BUG.
   * 
   * @param success para os testes em modo desenvolvimento exibir na console
   *        java as mensagens de debug quando a execuo  com sucesso.
   * 
   */
  public static void setDebugMode(boolean error, boolean success) {
    DEBUG_BUG = error;
    DEBUG_NORMAL = success;
  }
}