package tecgraf.javautils.gui.field;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.swing.text.BadLocationException;

import tecgraf.javautils.core.lng.LNG;

/**
 * Campo para Data. A mscara do campo  criada de acordo com o {@link Locale}
 * usado no construtor. O construtor default obtm o {@link Locale} do
 * {@link LNG}. Os <CODE>locales</CODE> permitidos so: pt_BR, es_AR e en_US.
 * Caso o {@link Locale} usado no construtor no seja um dos permitidos, o pt_BR
 * ser usado. O componente possui mtodos para validar e obter os valores da
 * data assim como setar a data atual.
 */
public class DateDocument extends RegexDocument {

  private static String sep = "/";

  public static enum Format {
    MMDDYYYY,
    DDMMYYYY;
  }

  private Format format;
  private Locale locale;

  /**
   * Constri um DateField com validao visual desabilitada. O {@link Locale}
   * usado ser obtido do {@link LNG}.
   * 
   */
  public DateDocument() {
    this(null);
  }

  /**
   * Constri um DateField para um determinado {@link Locale}.
   * 
   * @param locale {@link Locale}
   */
  public DateDocument(Locale locale) {
    super(getRegex(getFormat(locale)), buildAutoComplete(getFormat(locale)));
    setLocale(locale);
  }

  public void setLocale(Locale locale) {
    if (locale == null) {
      locale = LNG.getLocale();
    }
    this.locale = locale;
    format = getFormat(locale);
  }

  /**
   * Obtm o formato da mscara para um determinado {@link Locale}
   * 
   * @param locale {@link Locale}
   * @return {@link Format}
   */
  private static Format getFormat(Locale locale) {
    if (locale == null) {
      locale = LNG.getLocale();
    }

    if (locale.toString().equals("es_AR")) {
      return Format.DDMMYYYY;
    }
    else if (locale.toString().equals("en_US")) {
      return Format.MMDDYYYY;
    }
    else { // pt_BR
      return Format.DDMMYYYY;
    }
  }

  private static String getRegex(Format format) {
    String year = "([0-9]{1,4})?";
    String daymonth = "([0-9]{1,2})?";
    String sep = "(/)?";
    String regex = "";
    if (format == Format.MMDDYYYY || format == Format.DDMMYYYY) {
      regex = daymonth + sep + daymonth + sep + year;
    }
    return regex;
  }

  public boolean isValid() {
    if (getDate() != null) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se uma data est no formato vlido via expresso regular.
   * 
   * @param date String da data
   * @param format Formato da data
   * 
   * @return boolean
   */
  private static boolean validateDateWithRegex(String date, Format format) {
    String daymonth = "[0-9]{1,2}";
    String year = "[0-9]{4}";
    String regex = "";
    if (format == Format.MMDDYYYY || format == Format.DDMMYYYY) {
      regex = daymonth + sep + daymonth + sep + year;
    }
    if (date != null && date.matches(regex)) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se uma data est no formato vlido.
   * 
   * @param date String da data
   * @param locale Locale para a data
   * 
   * @return boolean
   */
  public static boolean validateDateFormat(String date, Locale locale) {
    Format format = getFormat(locale);
    return validateDateWithRegex(date, format);
  }

  /**
   * Obtm um {@link Date} referente a data ou <CODE>NULL</CODE> caso a data
   * esteja errada.
   * 
   * @param date Data em String
   * @param locale Locale em que se encontra a data
   * 
   * @return array de interger com a data
   */
  public static Date getDate(String date, Locale locale) {
    Format format = getFormat(locale);

    if (validateDateWithRegex(date, format)) {
      int year = 0;
      int month = 0;
      int day = 0;
      String[] split = date.split("/");
      if (format == Format.DDMMYYYY) {
        day = Integer.parseInt(split[0]);
        month = Integer.parseInt(split[1]);
        year = Integer.parseInt(split[2]);
      }
      else if (format == Format.MMDDYYYY) {
        month = Integer.parseInt(split[0]);
        day = Integer.parseInt(split[1]);
        year = Integer.parseInt(split[2]);
      }
      try {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setLenient(false);
        gc.clear();
        gc.set(GregorianCalendar.DAY_OF_MONTH, day);
        gc.set(GregorianCalendar.MONTH, month - 1);
        gc.set(GregorianCalendar.YEAR, year);
        return gc.getTime();
      }
      catch (Exception e) {
        return null;
      }
    }

    return null;
  }

  /**
   * Obtm o {@link GregorianCalendar} com a data do texto ou <CODE>NULL</CODE>
   * caso esteja preenchida errada.
   * 
   * @return {@link GregorianCalendar}
   */
  private GregorianCalendar getGregorianCalendar() {
    Long date = getDate();
    if (date != null) {
      GregorianCalendar gc = new GregorianCalendar();
      gc.setTimeInMillis(date);
      return gc;
    }
    return null;
  }

  /**
   * Obtm o dia ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getDay() {
    GregorianCalendar gc = getGregorianCalendar();
    if (gc != null) {
      return gc.get(GregorianCalendar.DAY_OF_MONTH);
    }
    return null;
  }

  /**
   * Obtm o ms ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getMonth() {
    GregorianCalendar gc = getGregorianCalendar();
    if (gc != null) {
      return gc.get(GregorianCalendar.MONTH) + 1;
    }
    return null;
  }

  /**
   * Obtm o ano ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getYear() {
    GregorianCalendar gc = getGregorianCalendar();
    if (gc != null) {
      return gc.get(GregorianCalendar.YEAR);
    }
    return null;
  }

  /**
   * Obtm a data em milisegundos ou NULL caso data seja invlida.
   * 
   * @return Long
   */
  public Long getDate() {
    try {
      String text = getText(0, getLength());
      Date date = getDate(text, locale);
      if (date != null) {
        return date.getTime();
      }
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Seta uma data.
   * 
   * @param day Dia
   * @param month Ms
   * @param year Ano
   * 
   */
  public void setDate(int day, int month, int year) {
    try {
      replace(0, getLength(), toString(format, day, month, year), null);
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
  }

  /**
   * Apaga o valor
   */
  public void clear() {
    try {
      remove(0, getLength());
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
  }

  /**
   * Seta uma data em milisegundos
   * 
   * @param lngDate
   */
  public void setDate(long lngDate) {
    try {
      replace(0, getLength(), toString(locale, lngDate), null);
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
  }

  /**
   * Seta a data atual
   */
  public void setToCurrentDate() {
    setDate(System.currentTimeMillis());
  }

  /**
   * Ativa o auto complemento. Complementa com o caracter do separador. Caso
   * seja o NULL nada acontece.
   * 
   * @param complete
   */
  private static CompleteText buildAutoComplete(final Format format) {
    if (format != null) {
      return new CompleteText() {
        public String buildCompletedText(String text, String newStr) {
          String regex1 = "";
          String regex2 = "";
          String complt1 = "";
          String complt2 = "";
          if (format == Format.DDMMYYYY || format == Format.MMDDYYYY) {
            regex1 = "[0-9]{2}";
            regex2 = "[0-9]{2}" + sep + "[0-9]{2}";
            complt1 = "[0-9]{1,2}(" + sep + "([0-9]{1,4})?)?";
            complt2 = "[0-9]{1,4}";
          }
          if ((text.matches(regex1) && newStr.matches(complt1))
            || text.matches(regex2) && newStr.matches(complt2)) {
            return sep + newStr;
          }
          return newStr;
        }
      };
    }
    else {
      return null;
    }
  }

  /**
   * Obtm uma {@link String} de data formatada
   * 
   * @param locale {@link Locale} usado para formatar a data
   * @param date Data em milisegundos
   * @return {@link String}
   */
  public static String toString(Locale locale, long date) {
    Format format = getFormat(locale);
    Calendar cal = new GregorianCalendar();
    cal.setTimeInMillis(date);
    int year = cal.get(Calendar.YEAR);
    int month = cal.get(Calendar.MONTH) + 1;
    int day = cal.get(Calendar.DAY_OF_MONTH);
    return toString(format, day, month, year);
  }

  /**
   * Obtm uma {@link String} de data formatada
   * 
   * @param format {@link Format} usado para formatar a data
   * @param day dia
   * @param month ms
   * @param year ano
   * @return {@link String}
   */
  public static String toString(Format format, int day, int month, int year) {
    String strDate = "";
    if (day > 0 && month > 0 && year > 0) {
      if (format == Format.DDMMYYYY) {
        strDate =
          getNumberStr(day) + sep + getNumberStr(month) + sep
            + getNumberStr(year);
      }
      else if (format == Format.MMDDYYYY) {
        strDate =
          getNumberStr(month) + sep + getNumberStr(day) + sep
            + getNumberStr(year);
      }
    }
    return strDate;
  }

  /**
   * Obtm o valor do nmero correspondente em Str. Caso o nmero seja 0 a 9 e o
   * separador seja o nulo, acrescenta 0 na frente para formar str com 2
   * caracteres.
   * 
   * @param value
   * 
   * @return String
   */
  private static String getNumberStr(int value) {
    if (value >= 0 && value <= 9) {
      return "0" + value;
    }
    return Integer.toString(value);
  }

  protected void rebuildCompleText() {
    setCompleteText(buildAutoComplete(getFormat(locale)));
  }

  @Override
  protected Object getValue() {
    return getDate();
  }

  protected Format getFormat() {
    return format;
  }

  @Override
  protected void fireAllListeners(Object oldValue, Object newValue,
    boolean valueIsAdjusting) {
    boolean monthChanged = false;
    Long oldDate = (Long) oldValue;
    Long newDate = (Long) newValue;

    if (oldValue != null && newValue != null) {
      int actualMonth;
      int actualYear;
      GregorianCalendar gc = new GregorianCalendar();
      gc.setTimeInMillis(oldDate);
      actualMonth = gc.get(GregorianCalendar.MONTH);
      actualYear = gc.get(GregorianCalendar.YEAR);
      gc.setTimeInMillis(newDate);
      boolean dateInSameMonth =
        (gc.get(GregorianCalendar.MONTH) == actualMonth);
      boolean dateInSameYear = (gc.get(GregorianCalendar.YEAR) == actualYear);
      monthChanged = dateInSameMonth && dateInSameYear;
    }
    DateEvent event = new DateEvent(newDate, monthChanged, valueIsAdjusting);

    DateListener[] listeners = listenerList.getListeners(DateListener.class);
    for (DateListener listener : listeners) {
      listener.dateUpdated(event);
    }
  }

  /**
   * Adiciona um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void addDateListener(DateListener listener) {
    listenerList.add(DateListener.class, listener);
  }

  /**
   * Remove um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void removeDateListener(DateListener listener) {
    listenerList.remove(DateListener.class, listener);
  }
}
