package tecgraf.javautils.gui.print;

import java.awt.Font;
import java.awt.print.PageFormat;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import javax.swing.table.DefaultTableModel;

/**
 * Exemplo de uso da classe {@link DefaultPrintableReport}. Retorna uma tabela
 * com ttulo, cabealho e ps-cabealho, contendo seis colunas, com dados
 * aleatrios. A ltima coluna  totalizada ao fim da tabela. A primeira coluna
 * apresenta uma fonte diferenciada, assim como os cabealhos de coluna e a
 * clula totalizadora.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DefaultPrintableReportSample2 {
  // *************************************
  // Constantes para os ndices de colunas
  // *************************************
  /** ndice da coluna de Navios */
  private static final int SHIP_COL = 0;
  /** ndice da coluna de Origens */
  private static final int SOURCE_COL = 1;
  /** ndice da coluna de ETD (Data Partida) */
  private static final int ETD_COL = 2;
  /** ndice da coluna de Destinos */
  private static final int TARGET_COL = 3;
  /** ndice da coluna de ETA (Data Chegada) */
  private static final int ETA_COL = 4;
  /** ndice da coluna de Volume */
  private static final int VOLUME_COL = 5;

  // *************************************
  // Constantes para os nomes de exemplo
  // *************************************
  /** Nomes de exemplo para navios */
  private static final Object[] SHIPS = new Object[] { "Norma", "Nilza",
      "Clara", "Pero", "Restinga", "TBN", "TBN", "TBN", "TBN" };
  /** Nomes de exemplo para pontos */
  private static final Object[] POINTS = new Object[] { "Ilha D'gua",
      "Angra dos Reis", "Mucuripe", "Cabedelo", "Santos", "N/A" };

  // *************************************
  // Constantes para as fontes utilizadas
  // *************************************
  /** Fonte para o ttulo da tabela */
  private static final Font TITLE_FONT = new Font("Serif", Font.BOLD, 14);
  /** Fonte para o cabealho da tabela, repetido a cada pgina */
  private static final Font TABLE_HEADER_FONT = new Font("SansSerif",
    Font.ITALIC, 8);
  /** Fonte para o texto a ser impresso ao fim da tabela */
  private static final Font POST_TABLE_FONT = new Font("SansSerif",
    Font.ITALIC, 8);
  /** Fonte para todas as colunas da tabela */
  private static final Font COL_NAME_FONT = new Font("SansSerif", Font.BOLD, 9);
  /** Fonte somente para a coluna "Navio" */
  private static final Font SHIP_FONT = new Font("SansSerif", Font.BOLD, 10);
  /** Fonte somente para a clula de "Volume Total" */
  private static final Font TOTAL_VOLUME_FONT = new Font("SansSerif",
    Font.BOLD, 9);

  /**
   * Mtodo de teste para a classe.
   * 
   * @param args no utilizados.
   * 
   * @throws PrinterException caso ocorra alguma exceo ao preparar a impresso
   *         do relatrio.
   */
  public static void main(String[] args) throws PrinterException {
    int numRows = 125;
    int numColumns = 6;

    DefaultTableModel model = getTableModel(numRows, numColumns);
    PrintableTable table = getPrintableTable(model);

    List<PrintableReportItem> page = new ArrayList<PrintableReportItem>();
    page.add(table);

    PrintableReport report = new DefaultPrintableReport(page);
    report.setOrientation(PageFormat.PORTRAIT);
    PrinterJob job = PrinterJob.getPrinterJob();
    job.setPageable(report);
    if (job.printDialog()) {
      job.print();
    }
  }

  /**
   * @param model modelo de dados para a tabela.
   * @return tabela para impresso.
   */
  public static PrintableTable getPrintableTable(DefaultTableModel model) {
    PrintableTable table = new DefaultPrintableTable(model);
    table.setTitle("Mapa de Navegao de Cabotagem\nViagens Programadas",
      TITLE_FONT);
    table.setTableHeader("Para alocao de petrleos.\nNo considera cabotagem"
      + " de derivados.", TABLE_HEADER_FONT);
    table.setPosText("Terminal N/A: indefinido.", POST_TABLE_FONT);

    int lastRow = model.getRowCount() - 1;
    int lastCol = model.getColumnCount() - 1;
    table.setColumnNameFont(0, lastCol, COL_NAME_FONT);
    table.setCellFont(0, lastRow, SHIP_COL, SHIP_COL, SHIP_FONT);
    table.setCellFont(lastRow, lastRow, VOLUME_COL, VOLUME_COL,
      TOTAL_VOLUME_FONT);

    table.setDefaultFont(new Font("SansSerif", Font.PLAIN, 9));
    table.setDecimalFormat(new DecimalFormat("#.00"));
    table.setLineWidth(0);
    table.setInsets(1, 1);
    return table;
  }

  /**
   * @return Navio aleatrio.
   */
  private static Object getRandomShip() {
    return SHIPS[(int) Math.floor(Math.random() * SHIPS.length)];
  }

  /**
   * @return Ponto aleatrio.
   */
  private static Object getRandomPoint() {
    return POINTS[(int) Math.floor(Math.random() * POINTS.length)];
  }

  /**
   * @return ltimo ponto da lista.
   */
  private static Object getLastPoint() {
    return POINTS[POINTS.length - 1];
  }

  /**
   * @return Volume aleatrio.
   */
  private static double getRandomVolume() {
    return Math.random() * 1000;
  }

  /**
   * @return ETD aleatria.
   */
  private static Date getRandomETD() {
    int days = (int) Math.floor((Math.random() - 0.5) * 30);
    Calendar cal = new GregorianCalendar();
    cal.add(Calendar.DAY_OF_MONTH, days);
    return cal.getTime();
  }

  /**
   * @return ETA aleatria.
   */
  private static Date getRandomETA() {
    int days = (int) Math.floor(Math.random() * 30);
    Calendar cal = new GregorianCalendar();
    cal.add(Calendar.DAY_OF_MONTH, days);
    return cal.getTime();
  }

  /**
   * @param numRows nmero desejado de linhas para a tabela.
   * @param numColumns nmero desejado de colunas para a tabela.
   * @return modelo para a tabela.
   */
  public static DefaultTableModel getTableModel(int numRows, int numColumns) {
    Object[] names =
      new Object[] { "Navio", "Origem", "ETD", "Destino", "ETA",
          "Volume\n(1000 m3)" };
    Object[][] data = new Object[numRows][numColumns];
    double totalVolume = 0;
    for (int row = 0; row < numRows - 1; row++) {
      data[row][SHIP_COL] = getRandomShip();
      data[row][SOURCE_COL] = getRandomPoint();
      data[row][TARGET_COL] = getRandomPoint();
      if (data[row][SOURCE_COL] == data[row][TARGET_COL]) {
        data[row][TARGET_COL] = getLastPoint();
      }
      data[row][ETD_COL] = getRandomETD();
      data[row][ETA_COL] = getRandomETA();
      double volume = getRandomVolume();
      data[row][VOLUME_COL] = volume;
      totalVolume += volume;
    }
    data[numRows - 1][SHIP_COL] = "Total";
    data[numRows - 1][VOLUME_COL] = totalVolume;
    DefaultTableModel m1 = new DefaultTableModel(data, names);
    return m1;
  }
}
