/*
 * $Id: LargeColumnPrintableTable.java 73492 2008-03-06 12:01:15Z letnog $
 */
package tecgraf.javautils.gui.print;

import java.awt.Font;
import java.util.BitSet;

import javax.swing.table.TableModel;

/**
 * Tabela a ser impressa que possui uma coluna muito grande. Esta coluna no
 * ser impressa como coluna, mas como uma nova linha abaixo da linha original.
 * 
 * As bordas horizontais das linhas sero impressas apenas abaixo da linha
 * com a coluna grande. S sero impressas as bordas verticais externas.
 */
public class LargeColumnPrintableTable extends DefaultPrintableTable {
  
  /**
   * Construtor.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param largeCol coluna muito grande a ser impressa como uma linha.
   */
  public LargeColumnPrintableTable(TableModel model, Font font, int largeCol) {
    this(model, font, largeCol, true, true);
  }
  
  /**
   * Construtor.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param largeCol coluna muito grande a ser impressa como uma linha.
   * @param changeCellFont indica se a fonte da clula pode ser alterada 
   *                       pela proporo.
   * @param changeTextFont indica se a fonte dos textos pode ser alterada 
   *                       pela proporo.
   */
  public LargeColumnPrintableTable(TableModel model, Font font, int largeCol,
                               boolean changeCellFont, boolean changeTextFont) {
    super(new LargeColumnModel(model, largeCol), font, changeCellFont, 
          changeTextFont);
    adjustTableFormat();
  }

  /**
   * Ajusta o formato da tabela. As bordas horizontais das linhas sero 
   * impressas apenas abaixo da linha com a coluna grande. S sero impressas 
   * as bordas verticais externas. A linha adicionada que corresponde  coluna
   * grande ter incio na segunda coluna e ser composta por todas as colunas
   * da linha.
   */
  private void adjustTableFormat() {
    LargeColumnModel model = LargeColumnModel.class.cast(getModel());
    int columnCount = model.getColumnCount();
    BitSet rows = model.getInsertedRows();
    for (int row=rows.nextSetBit(0); row>=0; row=rows.nextSetBit(row+1)) {
      joinColumns(row, LargeColumnModel.START_JOIN_COLUMN, columnCount-1);
      setLineBorderVisible(row-1, row, false);
    }
    setColumnBorderVisible(0, columnCount-1, false);
    setAlignTableHeader(true);
    setInsets(2, 1);
  }
  
  /**
   * {@inheritDoc}
   */
  @Override
  public void setCellFont(int fromRow, int toRow, int fromColumn, int toColumn,
                          Font font) {
    LargeColumnModel model = LargeColumnModel.class.cast(getModel());
    int fromViewCol = model.getViewColumnIndex(fromColumn);
    /* Coluna no apresentada */
    if (fromViewCol == -1) {
      fromViewCol = model.getViewColumnIndex(fromColumn + 1);
    }
    int toViewCol = model.getViewColumnIndex(toColumn);
    /* Coluna no apresentada */
    if (toViewCol == -1) {
      toViewCol = model.getViewColumnIndex(toColumn - 1);
    }
    int fromViewRow = model.getViewRowIndex(fromRow);
    int toViewRow = model.getViewRowIndex(toRow);
    super.setCellFont(fromViewRow, toViewRow, fromViewCol, toViewCol, font);
  }
  
  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnPart(float[] columnPart) {
    LargeColumnModel model = LargeColumnModel.class.cast(getModel());
    float[] parts = new float[model.getColumnCount()];
    for (int col=0; col<columnPart.length; col++) {
      int viewCol = model.getViewColumnIndex(col);
      if (viewCol == -1) {
        continue;
      }
      parts[viewCol] = columnPart[col];
    }
    super.setColumnPart(parts);
  }
}
