/* 
 * $Author$
 * $Date$
 * $Revision: 108627 $
 */
package tecgraf.javautils.gui.tree;

import javax.swing.JFrame;
import javax.swing.WindowConstants;

import tecgraf.javautils.gui.tree.Node.FiltrageMode;


/**
 * Exemplo de uso da classe {@link FilterableTreePanel}.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class FilterableTreePanelSample {

  /**
   * Inicia o programa.
   * 
   * @param args Cada elemento do array representa a quantidade de elementos de
   *        um nvel. O primeiro elemento  o primeiro nvel (filhos da raz), o
   *        segundo elemento  o segundo nvel (netos da raz) e assim
   *        sucessivamente (No aceita {@code null}).
   */
  public static void main(String[] args) {
    int[] childCountsPerLevel = new int[args.length];
    for (int argIx = 0; argIx < args.length; argIx++) {
      int childCount;
      try {
        childCount = Integer.parseInt(args[argIx]);
        childCountsPerLevel[argIx] = childCount;
        if (childCount < 0) {
          showUsage();
          System.exit(1);
        }
      } catch (NumberFormatException e) {
        showUsage();
        System.exit(1);
      }
    }

    DefaultNode rootNode = createTree(childCountsPerLevel);

    FilterableTreePanel panel = new FilterableTreePanel(rootNode);
    JFrame frame = new JFrame("Exemplo de rvore filtrvel");
    frame.add(panel);
    frame.pack();
    frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
    frame.setVisible(true);
  }

  /**
   * Cria o rtulo de um n.
   * 
   * @param parentNode O pai do n (No aceita {@code null}).
   * @param levelIndex O ndice do nvel.
   * @param childIndex O ndice do filho.
   * 
   * @return .
   */
  private static String createLabel(DefaultNode parentNode, int levelIndex,
    int childIndex) {
    String parentLabel = parentNode.getLabel();
    parentLabel = parentLabel.replaceAll("[0-9]:", "");
    String separator = (parentLabel.length() == 0 ? "" : parentLabel + ".");
    return (levelIndex + 1) + ":" + separator + (childIndex + 1);
  }

  /**
   * Cria a rvore.
   * 
   * @param parentNode O pai do nvel que ser criado (No aceita {@code null}).
   * 
   * @param childCountsPerLevel O array com o nmero de ns por nvel (No
   *        aceita {@code null}).
   * @param levelIndex O ndice do nvel corrente.
   */
  private static void createTree(DefaultNode parentNode,
    int[] childCountsPerLevel, int levelIndex) {
    int childCount = childCountsPerLevel[levelIndex];
    for (int childIndex = 0; childIndex < childCount; childIndex++) {
      FiltrageMode mode = getFiltrageMode(levelIndex, childIndex,
        childCountsPerLevel.length - 1);
      DefaultNode childNode = new DefaultNode(createLabel(parentNode,
        levelIndex, childIndex), mode);
      int nextLevelIndex = levelIndex + 1;
      if (nextLevelIndex < childCountsPerLevel.length) {
        createTree(childNode, childCountsPerLevel, nextLevelIndex);
      }
      parentNode.addChild(childNode);
    }
  }

  /**
   * Obtm o modo de filtragem para o n que ser criado.
   * 
   * @param levelIndex O ndice do nvel.
   * @param nodeIndex O ndice do n no nvel.
   * @param maxLevelIndex O ndice mximo de nveis. 
   * 
   * @return .
   */
  private static FiltrageMode getFiltrageMode(int levelIndex, int nodeIndex,
    int maxLevelIndex) {
    if (maxLevelIndex == levelIndex) {
      FiltrageMode[] modes = FiltrageMode.values();
      return modes[nodeIndex % modes.length];
    }
    return FiltrageMode.EVALUATE;
  }

  /**
   * Cria a rvore.
   * 
   * @param childCountsPerLevel O array com o nmero de ns por nvel (No
   *        aceita {@code null}).
   * 
   * @return a raz da rvore.
   * 
   */
  private static DefaultNode createTree(int[] childCountsPerLevel) {
    DefaultNode rootNode = new DefaultNode("0:");
    if (childCountsPerLevel.length > 0) {
      createTree(rootNode, childCountsPerLevel, 0);
    }
    return rootNode;
  }

  /**
   * Exibe informaes sobre o uso deste programa.
   */
  private static void showUsage() {
    String programCall = 
      String.format("java %s", FilterableTreePanelSample.class.getName());
    StringBuilder usage = new StringBuilder();
    usage.append("Uso:");
    usage.append("\n");
    usage.append("\t");
    usage.append(programCall);
    usage.append(" 0..*[<quantidade_de_ns_por_nvel>]");
    usage.append("\n");
    usage.append("\n");
    usage.append("\t");
    usage.append(
      "quantidade_de_ns_por_nvel: nmero de ns em cada nvel da rvore.");
    usage.append("\n");
    usage.append("\n");
    usage.append("Exemplo:");
    usage.append("\n");
    usage.append("\t");
    usage.append(programCall);
    usage.append(" 2 3 4");
    usage.append("\n");
    usage.append("\n");
    usage.append("\t");
    usage.append("Cria uma rvore com 4 nveis:");
    usage.append("\n");
    usage.append("\t");
    usage.append("* Nvel 1: raz oculta.");
    usage.append("\n");
    usage.append("\t");
    usage.append("* Nvel 2: 2 elementos.");
    usage.append("\n");
    usage.append("\t");
    usage.append("* Nvel 3: 3 elementos.");
    usage.append("\n");
    usage.append("\t");
    usage.append("* Nvel 4: 4 elementos.");
    usage.append("\n");
    System.err.println(usage);
  }
}
