package busexplorer.utils;

import busexplorer.Application;
import busexplorer.desktop.dialog.ExceptionDialog;
import busexplorer.exception.BusExplorerHandlingException;
import busexplorer.exception.handling.ExceptionContext;
import busexplorer.exception.handling.ExceptionHandler;
import tecgraf.javautils.gui.Task;

import java.awt.Dialog.ModalityType;


/**
 * A classe abstrata BusExplorerTask permite a realizao das tarefas do
 * BusExplorer em threads separadas. Ela realiza o tratamento padro de excees
 * para tais tarefas.
 * 
 * @author Tecgraf
 * @param <T> A classe do resultado da tarefa.
 */
public abstract class BusExplorerTask<T> extends Task<T> {

  /** Manipulador de excees */
  private ExceptionHandler<BusExplorerHandlingException> handler;
  /** Contexto das excees recebidas */
  private ExceptionContext context;

  /**
   * Construtor para garantir o uso do {@link ExceptionHandler}
   * definido na {@link Application#exceptionHandler()}.
   *
   * @param context Contexto das excees recebidas.
   */
  public BusExplorerTask(ExceptionContext context) {
    this.handler = Application.exceptionHandler();
    this.context = context;
  }

  /**
   * Construtor para permitir a definio de um {@link ExceptionHandler}
   * diferente do padro definido na {@link Application#exceptionHandler()} e tambm
   * permite configurar o modal da janela, com uso do construtor {@link Task#Task(ModalityType)}.
   * 
   * @param modality Tipo de modalidade do dilogo.
   * @param handler Manipulador de excees.
   * @param context Contexto das excees recebidas.
   */
  public BusExplorerTask(ExceptionContext context,
                         ExceptionHandler<BusExplorerHandlingException> handler,
                         ModalityType modality) {
    super(modality);
    this.context = context;
    this.handler = handler;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleError(Exception exception) {
    BusExplorerHandlingException handlingException =
      handler.process(exception, context);
    ExceptionDialog.createDialog(parentWindow, taskTitle,
      handlingException.getException(),
      handlingException.getErrorMessage()).setVisible(true);
  }

  /**
   * Tarefa a ser executada. Esse mtodo deve ser implementado para executar a
   * tarefa.
   *
   * @throws Exception quando a ocorrncia de um erro (representado pela
   *         exceo) impediu a realizao da tarefa
   *
   * @see Task#performTask()
   */
  protected abstract void doPerformTask() throws Exception;

  /**
   * Mtodo para estender o tratamento de erros da {@link Task#performTask()} que no
   * fornece tratamento para erros derivados de {@link java.lang.Error}.
   * <p>
   * Para contornar essa limitao, e poder apresentar os detalhes ao usurio,  lanado
   * um {@link RuntimeException} com o erro original aninhado.
   *
   * @throws Exception a exceo original caso o {@link BusExplorerTask#doPerformTask()}
   *         tenha lanado uma exceo.
   * @throws RuntimeException uma exceo de runtime contendo o {@link java.lang.Error}
   *         caso o {@link BusExplorerTask#doPerformTask()} tenha lanado um erro.
   *
   * @see Task#performTask()
   */
  protected final void performTask() throws Exception {
    try {
      this.doPerformTask();
    } catch (Error e) {
      throw new RuntimeException(e);
    }
  }
}
