package migration.dao;

import migration.utils.Log;
import migration.utils.Utils;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import java.sql.*;
import java.util.List;
import java.util.Properties;

public class JobDAO {
  private Connection connection;
  private Properties sql;
  private Properties config;

  public JobDAO(Properties config) {
    this.config = config;
    this.sql = Utils.getConfigPropertiesFromResource("sql.properties");
  }

  public Connection getConn(){
    try {
      if (this.connection == null || this.connection.isClosed()) {
        Log.print("Getting a new DB CONNECTION!");
        Properties connConfig = new Properties();
        connConfig.put("user", this.config.getProperty("dbUser"));
        connConfig.put("password", this.config.getProperty("dbPassword"));
        this.connection = DriverManager.getConnection(this.config.getProperty("dbHost"), connConfig);
      }
    } catch (SQLException e) {
      e.printStackTrace();
      System.exit(1);
    }
    return this.connection;
  }

  private boolean insertJobAlgorithms(List<JSONObject> algorithms, JSONArray nodes, int jobId) {
    try {
      for (JSONObject algorithm : algorithms) {
        int jobAlgorithmId = getNextSequenceId("job_algorithms_seq");
        PreparedStatement st = getConn().prepareStatement(sql.getProperty("insert_job_algorithms"));
        st.setInt(1, jobAlgorithmId);
        st.setString(2, algorithm.getString("id"));
        st.setString(3, algorithm.getString("name"));
        st.setString(4, algorithm.getString("version"));
        if (algorithm.has("flowNodeId")) {
          int flowNodeId = algorithm.getInt("flowNodeId");
          st.setInt(5, flowNodeId);
          Integer flowNodeExitCode = getNodeExitCode(nodes, flowNodeId);
          if (flowNodeExitCode != null) {
            st.setInt(7, flowNodeExitCode);
          } else {
            st.setNull(7, Types.INTEGER);
          }
        } else {
          st.setInt(5, 0);
          st.setNull(7, Types.INTEGER);
        }
        st.setInt(6, jobId);
        st.executeUpdate();
        st.close();
        if (algorithm.has("params")) {
          if (!insertJobAlgorithmParameters(algorithm, jobAlgorithmId)) {
            return false;
          }
        }
      }
      return true;
    } catch (SQLException | JSONException e) {
      Log.print(e.toString());
      return false;
    }
  }

  private Integer getNodeExitCode(JSONArray nodes, Integer nodeId) {
    try {
      for (int i = 0; i < nodes.length(); i++) {
        JSONObject node = nodes.getJSONObject(i);
        if (node.getInt("flow_node_id") == (nodeId)) {
          if (node.has("flow_node_exit_code")) {
            return node.getInt("flow_node_exit_code");
          } else {
            return null;
          }
        }
      }
    } catch (JSONException e) {
      Log.print(e.toString());
      return null;
    }

    return null;
  }

  private Integer getGuiltyNode(JSONArray nodes) {
    try {
      for (int i = 0; i < nodes.length(); i++) {
        JSONObject node = nodes.getJSONObject(i);
        if (node.has("flow_node_is_guilty") && node.getBoolean("flow_node_is_guilty")) {
          if (node.has("flow_node_id")) {
            return node.getInt("flow_node_id");
          }
        }
      }
    } catch (JSONException e) {
      Log.print(e.toString());
      return null;
    }

    return null;
  }

  private boolean insertJobAlgorithmParameters(JSONObject algorithm, int jobAlgorithmId) {
    try{
      JSONArray algorithmParameters = algorithm.getJSONArray("params");
      for (int i = 0; i < algorithmParameters.length(); i++) {
        JSONObject param = algorithmParameters.getJSONObject(i);
        int jobAParamAlgorithmId = getNextSequenceId("job_algo_params_seq");
        PreparedStatement st = getConn().prepareStatement(sql.getProperty("insert_job_algo_params"));
        st.setInt(1, jobAParamAlgorithmId);
        st.setString(2, param.getString("param_label"));
        st.setString(3, param.getString("param_id"));
        st.setString(4, param.getString("param_type"));
        st.setInt(5, jobAlgorithmId);
        st.executeUpdate();
        st.close();
        if(!insertJobAlgorithmParameterValues(param, jobAParamAlgorithmId)){
          return false;
        }
      }
    } catch (SQLException | JSONException e) {
      Log.print(e.toString());
    }
    return true;
  }

  private boolean insertJobAlgorithmParameterValues(JSONObject param, int jobAParamAlgorithmId) {
    try{
      PreparedStatement st = getConn().prepareStatement(sql.getProperty("insert_job_param_values"));
      int jobAParamId = getNextSequenceId("job_param_values_seq");
      st.setInt(1, jobAParamAlgorithmId);
      st.setInt(2, jobAParamId);
      st.setString(3, param.getString("param_value"));
      st.executeUpdate();
      st.close();
      return true;
    } catch (SQLException | JSONException e) {
      Log.print(e.toString());
      return false;
    }
  }

  private boolean insertStatusHistoryFromJob(JSONObject jobData, int jobId) {
    try{
      JSONArray statusListJSON = jobData.getJSONArray("statusHistory");
      for (int i = 0; i < statusListJSON.length(); i++) {
        try{
          int statusId = getNextSequenceId("job_status_history_seq");
          JSONObject status = statusListJSON.getJSONObject(i);
          Timestamp statusTimestamp = null;
          if (status.has("timestamp")) {
            statusTimestamp = Utils.parseDate(status.getString("timestamp"), config.getProperty("timezoneAPICSBASE"));
          } else {
            if (status.getString("status").toLowerCase().equals("finished")) {
              statusTimestamp = Utils.parseDate(jobData.getString("endTime"), config.getProperty("timezoneAPICSBASE"));
            }
          }
          if(statusTimestamp!=null) {
            PreparedStatement st = getConn().prepareStatement(sql.getProperty("insert_job_status_history"));
            st.setInt(1, statusId);
            st.setString(2, status.getString("status"));
            st.setTimestamp(3, statusTimestamp);
            st.setInt(4, jobId);
            st.executeUpdate();
            st.close();
            getConn().commit();
          }
        } catch (SQLException e) {
          getConn().rollback();
          Log.print(e.toString());
        }
      }
      return true;
    } catch (SQLException | JSONException e) {
      Log.print(e.toString());
      return false;
    }
  }

  private boolean insertJobRow(JSONObject jobData, int jobId, Integer guiltyNodeId) {
    try {
      PreparedStatement st = getConn().prepareStatement(sql.getProperty("insert_job"));
      st.setInt(1, jobId); // id
      st.setString(2, jobData.getString("jobId")); // job_id
      st.setString(3, jobData.getString("sessionId")); // project_id
      st.setString(4, jobData.getString("jobOwner")); // user_id
      st.setTimestamp(5, Utils.parseDate(jobData.getString("submissionTime"), config.getProperty("timezoneAPICSBASE"))); // submission_time
      st.setString(6, jobData.getString("executionMachine")); // execution_machine
      if(jobData.has("endTime") && !jobData.getString("endTime").isEmpty()){
        st.setTimestamp(7, Utils.parseDate(jobData.getString("endTime"), config.getProperty("timezoneAPICSBASE"))); // end_time
      } else {
        st.setNull(7, Types.TIMESTAMP); // end_time
      }
      st.setInt(8, Integer.parseInt(jobData.getString("numberOfAttempts"))); // number_of_retries
      if(jobData.has("description") && !jobData.getString("description").isEmpty()) {
        st.setString(9, jobData.getString("description")); // description
      } else {
        st.setNull(9, Types.VARCHAR); // description
      }
      st.setInt(10, Integer.parseInt(jobData.getString("priority"))); // priority
      if (jobData.has("exitStatus")) {
        st.setString(11, jobData.getString("exitStatus")); // exit_status
      } else {
        st.setNull(11, Types.VARCHAR); // exit_status
      }
      if (jobData.has("wallclockTime")) {
        st.setInt(12, Integer.parseInt(jobData.getString("wallclockTime"))); // wall_clock_time
      } else {
        st.setNull(12, Types.INTEGER); // wall_clock_time
      }
      st.setTimestamp(13, Utils.parseDate(jobData.getString("lastModifiedTime"), config.getProperty("timezoneAPICSBASE"))); // last_modified_time
      st.setBoolean(14, jobData.getBoolean("automaticallyMachineSelection")); // auto_machine_selected
      st.setBoolean(15, false); // is_deleted - se um job é retornado pela API REST, não foi deletado (?) default discutido com equipe
      st.setString(16, jobData.getString("jobType")); // job_type
      st.setBoolean(17, false); // multiple_execution ? default discutido com equipe
      st.setInt(18, 1); // number_of_processes ? default discutido com equipe
      st.setInt(19, 1);// number_of_processes_by_machine ? default discutido com equipe
      st.setString(20, jobData.getString("groupId")); // group_id
      if (jobData.has("exitCode")) {
        st.setInt(21, jobData.getInt("exitCode")); // exit_code
      } else {
        st.setNull(21, Types.INTEGER); // exit_code
      }
      if (jobData.has("flowRaw")) {
        st.setBytes(22, jobData.getString("flowRaw").getBytes()); // flow_raw
      }
      else {
        st.setNull(22, Types.BINARY); // flow_raw
      }
      if (guiltyNodeId != null) {
        st.setInt(23, guiltyNodeId); // guilty_node_id
      } else {
        st.setNull(23, Types.INTEGER); // guilty_node_id
      }
      st.executeUpdate();
      st.close();
      return true;
    }
    catch (SQLException | JSONException e) {
      try {
        Log.print(e.toString());
        getConn().rollback();
      }
      catch (SQLException ex) {
        ex.printStackTrace();
      }
    }
    return false;
  }

  public boolean insertJob(JSONObject jobData, List<JSONObject> algorithms, JSONArray nodes){
    try {
      int jobId = getNextSequenceId("jobs_seq");
      getConn().setAutoCommit(false);
      if (insertJobRow(jobData, jobId, getGuiltyNode(nodes))) {
        if (insertJobAlgorithms(algorithms, nodes, jobId)) {
          getConn().commit();
          insertStatusHistoryFromJob(jobData, jobId);
          return true;
        }
      }
      getConn().rollback();
      return false;
    } catch (SQLException e) {
        try {
          Log.print(e.toString());
          getConn().rollback();
        } catch (SQLException ex) {
          Log.print(ex.toString());
        }
      return false;
    }
  }

  private int getNextSequenceId(String sequenceName){
    int nextSequenceNumber = 0;
    try {
      PreparedStatement st = getConn().prepareStatement(sql.getProperty("query_next_sequence_number"));
      st.setString(1, sequenceName);
      ResultSet rs = st.executeQuery();
      while (rs.next()){
        nextSequenceNumber = rs.getInt(1);
      }
      rs.close();
      st.close();
      return nextSequenceNumber;
    } catch (SQLException e) {
      e.printStackTrace();
      System.exit(1);
    }
    return nextSequenceNumber;
  }

  public boolean jobExists(String jobId) {
    try {
      PreparedStatement st = getConn().prepareStatement(sql.getProperty("check_job_exist"));
      st.setString(1, jobId);
      ResultSet rs = st.executeQuery();
      return rs.next();
    } catch (SQLException e) {
      e.printStackTrace();
      System.exit(1);
    }

    return false;
  }
}
