package br.pucrio.tecgraf.soma.serviceapi.persistence.repository.impl;

import java.util.List;

import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Repository;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import jakarta.persistence.EntityManager;
import jakarta.persistence.TypedQuery;
import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;
import jakarta.transaction.Transactional;

@Transactional
public abstract class JPARepository<T> implements Repository<T, JPASpecification<T>> {

  @Override
  public void add(T element) {
    getEntityManager().persist(element);
  }

  @Override
  public void remove(T element) {
    getEntityManager().remove(element);
  }

  @Override
  public void update(T element) {
    getEntityManager().flush();
  }

  private TypedQuery<T> toQuery(JPASpecification<T> specification, Sort... sorts) {
    CriteriaBuilder criteriaBuilder = getEntityManager().getCriteriaBuilder();
    CriteriaQuery<T> criteriaQuery = criteriaBuilder.createQuery(getType());
    Root<T> root = criteriaQuery.from(getType());
    Predicate predicate = specification.toPredicate(root, criteriaBuilder);
    criteriaQuery.where(predicate);
    for (Sort sort : sorts) {
      if (sort.isAscending()) {
        criteriaQuery.orderBy(criteriaBuilder.asc(root.get(sort.getAttribute())));
      }
      else {
        criteriaQuery.orderBy(criteriaBuilder.desc(root.get(sort.getAttribute())));
      }
    }
    return getEntityManager().createQuery(criteriaQuery);
  }

  @Override
  public List<T> find(JPASpecification<T> specification, Sort... sorts) {
    return find(specification, null, null, sorts);
  }

  @Override
  public List<T> find(JPASpecification<T> specification, Integer maxResult, Sort... sorts) {
    return find(specification, maxResult, null, sorts);
  }

  @Override
  public List<T> find(JPASpecification<T> specification, Integer maxResult, Integer offset, Sort... sorts) {
    TypedQuery<T> query = toQuery(specification, sorts);
    if (maxResult != null) {
      if (maxResult <= 0) {
        throw new IllegalArgumentException();
      }
      query.setMaxResults(maxResult);
    }
    if (offset != null) {
      if (offset < 0) {
        throw new IllegalArgumentException();
      }
      query.setFirstResult(offset);
    }
    return query.getResultList();
  }

  @Override
  public long count(JPASpecification<T> specification) {
    CriteriaBuilder criteriaBuilder = getEntityManager().getCriteriaBuilder();
    CriteriaQuery<Long> criteriaQuery = criteriaBuilder.createQuery(Long.class);
    Root<T> root = criteriaQuery.from(getType());
    criteriaQuery.select(criteriaBuilder.count(root));
    Predicate predicate = specification.toPredicate(root, criteriaBuilder);
    criteriaQuery.where(predicate);
    long result = getEntityManager().createQuery(criteriaQuery).getSingleResult();
    return result;
  }

  @Override
  public T first(JPASpecification<T> specification, Sort... sorts) {
    try {
      return toQuery(specification, sorts).setMaxResults(1).getSingleResult();
    }
    catch (jakarta.persistence.NoResultException e) {
      return null;
    }
  }

  public abstract Class<T> getType();

  public abstract EntityManager getEntityManager();
}
