package br.pucrio.tecgraf.soma.job.menu;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class CommandLineMenu {

  private Options options;

  private Map<String, Boolean> unique;
  private Map<String, String> mandatory;
  private Map<String, String> optional;
  private Map<String, String> mapDefault;

  private boolean parsed;

  private void startValues() {
    this.unique = new HashMap<String, Boolean>();
    this.mandatory = new HashMap<String, String>();
    this.optional = new HashMap<String, String>();
    this.mapDefault = new HashMap<String, String>();
    this.parsed = false;
  }

  public CommandLineMenu(Options options) {
    this.options = options;
    startValues();
  }

  public CommandLineMenu() {
    this.options = new Options();
    startValues();
  }

  public Options getOptions() {
    return this.options;
  }

  /**
   * Adds types in maps
   * 
   * @param type option type (e.g unique)
   * @param name name of the option
   * @param value value of the option
   */
  private void addType(OptionType type, String name, String value) {
    if (OptionType.UNIQUE.equals(type)) {
      this.unique.put(name, false);
    }
    else if (OptionType.MANDATORY.equals(type)) {
      this.mandatory.put(name, value);
    }
    else if (OptionType.OPTIONAL.equals(type)) {
      this.optional.put(name, value);
    }
    else if (OptionType.DEFAULT.equals(type)) {
      this.mapDefault.put(name, value);
    }
  }

  /**
   * Creates an Option.Builder with this argument (if there is)
   * 
   * @param shortName short name
   * @param longName long name
   * @param description description
   * @return return option
   */
  public Option.Builder createOptionBuilder(String shortName, String longName, String description) {
    Option.Builder opBuilder;
    if (shortName == null || shortName.equals("")) {
      opBuilder = Option.builder();
    }
    else {
      opBuilder = Option.builder(shortName);
    }
    opBuilder = opBuilder.longOpt(longName).desc(description);
    return opBuilder;
  }

  private void addOption(String shortName, String longName, String description, String argName) {
    Option.Builder opBuilder = createOptionBuilder(shortName, longName, description);
    opBuilder = addArg(opBuilder, argName);
    this.options.addOption(opBuilder.build());
  }

  /**
   * Add option.
   * 
   * @param shortName short name. e.g: v
   * @param longName long name. e.g: version
   * @param description description
   * @param argName argument name
   * @param type
   */
  public void addOption(String shortName, String longName, String description, String argName, OptionType type) {
    addOption(shortName, longName, description, argName);
    addType(type, longName, null);
  }

  /**
   * Add option.
   * 
   * @param shortName short name. e.g: v
   * @param longName long name. e.g: version
   * @param description description
   * @param argName argument name
   * @param type
   */
  public void addOption(String shortName, String longName, String description, String argName, OptionType type,
    String value) {
    if (type.equals(OptionType.DEFAULT)) {
      addOption(shortName, longName, description, argName);
      addType(type, longName, value);
    }
  }

  /**
   * Add option.
   * 
   * @param shortName short name. e.g: v
   * @param longName long name. e.g: version
   * @param description description
   */
  public void addOption(String builder, String longName, String description, OptionType type) {
    this.options.addOption(createOptionBuilder(builder, longName, description).build());
    addType(type, longName, null);
  }

  /**
   * Add argument
   * 
   * @param opBuilder Option builder
   * @param argName argument name
   * @return option builder with the argument
   */
  private Option.Builder addArg(Option.Builder opBuilder, String argName) {
    return opBuilder.hasArg().argName(argName);
  }

  private void addValue(CommandLine line, Map<String, String> map) {
    for (String key : map.keySet()) {
      if (line.hasOption(key)) {
        map.put(key, line.getOptionValue(key));
      }
    }
  }

  private int calcSizeWithoutNullValue(Map<String, String> map) {
    int count = 0;
    for (String key : map.keySet()) {
      if (map.get(key) != null) {
        ++count;
      }
    }
    return count;
  }

  /**
   * Parse arguments and insert in the maps
   * 
   * @param args arguments from command line
   * @throws ParseException
   */
  public void parser(String[] args) throws ParseException {
    CommandLineParser parser = new DefaultParser();
    CommandLine line = parser.parse(this.options, args);
    addValueUnique(line, this.unique);
    addValue(line, this.mandatory);
    addValue(line, this.optional);
    addValue(line, this.mapDefault);
    this.parsed = true;
  }

  private void addValueUnique(CommandLine line, Map<String, Boolean> unique) {
    for (String key : unique.keySet()) {
      if (line.hasOption(key)) {
        unique.put(key, true);
      }
    }
  }

  /**
   * verify unique element
   * 
   * @return if has unique element, then return true.
   */
  public boolean hasUnique() {
    for (String key : this.unique.keySet()) {
      if (this.unique.get(key) == false) {
        this.unique.remove(key);
      }
    }
    return this.parsed == true && this.unique.size() == 1;
  }

  private void checkParseState() throws Exception {
    if (!this.parsed) {
      throw new Exception("Call parser before get unique element");
    }
  }

  /**
   * Get unique element
   * 
   * @return unique element (e.g: version or help)
   * @throws Exception
   */
  public String getUnique() throws Exception {
    checkParseState();
    if (this.unique.size() > 1) {
      String exceptionMsg = "You can not use ";
      final int size = this.unique.size();
      int i = 1;
      for (String key : this.unique.keySet()) {
        if (size < i) {
          exceptionMsg += this.unique.get(key) + ", ";
        }
        else {
          exceptionMsg += this.unique.get(key);
        }
        ++i;
      }
      exceptionMsg += " at the same time";
      throw new Exception(exceptionMsg);
    }
    else if (this.unique.size() == 0) {
      throw new Exception("There is not unique option");
    }
    for (String key : this.unique.keySet()) {
      return key;
    }
    return null;
  }

  /**
   * Verify if has all mandatory elements
   * 
   * @return if has all mandatory elements, then return true
   */
  public boolean hasAllMandatory() {
    return (this.parsed == true) && (calcSizeWithoutNullValue(this.mandatory) == this.mandatory.size());
  }

  public Map<String, String> getMandatory() throws Exception {
    checkParseState();
    if (hasAllMandatory()) {
      return this.mandatory;
    }
    String expectionMsg = "";
    int erroCount = 0;
    for (String key : this.mandatory.keySet()) {
      if (this.mandatory.get(key) == null) {
        ++erroCount;
        if (erroCount > 1) {
          expectionMsg += ", ";
        }
        expectionMsg += key;
      }
    }
    if (erroCount == 1) {
      expectionMsg = expectionMsg + " is mandatory\n";
    }
    else if (erroCount > 1) {
      expectionMsg = "They are mandatory: " + expectionMsg + "\n";
    }
    throw new Exception(expectionMsg);
  }

  /**
   * Verify if there are optional and default values
   * 
   * @return if has optional and default values, then return true
   */
  public boolean hasOptionalAndDefault() {
    try {
      return this.parsed == true && getOptionalAndDefault().size() >= 1;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Get optional and default values
   * 
   * @return return map with optional and default elements
   * @throws Exception
   */
  public Map<String, String> getOptionalAndDefault() throws Exception {
    checkParseState();
    Map<String, String> result = new HashMap<String, String>();
    for (String key : this.optional.keySet()) {
      final String value = this.optional.get(key);
      if (value != null) {
        result.put(key, value);
      }
    }
    for (String key : this.mapDefault.keySet()) {
      result.put(key, this.mapDefault.get(key));
    }
    return result;
  }

}
