package br.pucrio.tecgraf.soma.logsmonitor.flow;

import br.pucrio.tecgraf.soma.logsmonitor.manager.WebSocketSessionManager;
import br.pucrio.tecgraf.soma.logsmonitor.model.Notification;
import br.pucrio.tecgraf.soma.logsmonitor.model.Topic;
import br.pucrio.tecgraf.soma.logsmonitor.model.mapper.TopicEventMapper;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.service.TopicService;
import br.pucrio.tecgraf.soma.logsmonitor.service.TopicServiceFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Flow;

/*
O Uso do escopo "prototype" faz com que o Spring crie uma nova instância dessa classe a cada vez que fizer a injeção
dela. E esse é o comportamento desejado ao chamar o método PublisherManager#subscribe.

Mais detalhes em:
https://docs.spring.io/spring-framework/docs/5.3.3/reference/html/core.html#beans-factory-scopes-prototype
 */
@Component
@Scope("prototype")
public class TopicSubscriber implements Flow.Subscriber<ResourceMonitorEvent> {
  private static final Log logger = LogFactory.getLog(TopicSubscriber.class);
  private final String sessionId;
  private final String subscriptionId;
  private final Topic topic;
  private Long currentSeqnum;
  private final TopicEventMapper mapper;
  @Autowired private TopicServiceFactory serviceFactory;
  @Autowired ObjectMapper objectMapper;
  @Autowired WebSocketSessionManager webSocketSessionManager;
  private Flow.Subscription subscription;

  // Flag para indicar que não há mais eventos a receber
  private Boolean completed;

  public TopicSubscriber(
      String sessionId,
      String subscriptionId,
      Topic topic,
      TopicEventMapper mapper,
      Long initialSeqnum) {
    logger.debug(
        String.format(
            "Topic [%s]: new TopicSubscriber with session [%s] and subscriptionId [%s]",
            topic, sessionId, subscriptionId));
    this.sessionId = sessionId;
    this.subscriptionId = subscriptionId;
    this.topic = topic;
    this.currentSeqnum = initialSeqnum;
    this.mapper = mapper;
    this.completed = false;
  }

  public String getSessionId() {
    return sessionId;
  }

  public String getSubscriptionId() {
    return subscriptionId;
  }

  @Override
  synchronized public void onSubscribe(Flow.Subscription subscription) {
    logger.debug(
        String.format(
            "Session [%s]: onSubscribe activated for topic [%s] with subscriptionId [%s]",
            sessionId, topic, subscriptionId));
    this.subscription = subscription;

    // Caso a flag esteja ativa pode cancelar a assinatura. Isso pode ocorrer caso a chamada do onComplete seja feita
    // logo em seguida ao cadastro de assinatura
    if (!completed) {
      subscription.request(1);
    } else {
      subscription.cancel();
    }
  }

  /**
   * Callback para notificar mudança no recurso monitorado. O evento contém as informações sobre a
   * alteração.
   *
   * @param event o evento notificado pelo monitor de recurso
   */
  @Override
  synchronized public void onNext(ResourceMonitorEvent event) {
    logger.debug(
        String.format(
            "Session [%s]: onNext activated for topic [%s] with subscriptionId [%s]",
            sessionId, topic, subscriptionId));
    WebSocketSession session = webSocketSessionManager.getSession(sessionId);
    if (session == null) {
      subscription.cancel();
      return;
    }
    try {
      if (session.isOpen()) {
        TopicService service = this.serviceFactory.getServiceByTopicType(this.topic.getTopicType());

        List<ResourceMonitorEvent> events = new ArrayList<>();
        if (currentSeqnum < event.getStartSeqnum()) {
          logger.debug(
              String.format(
                  "The subscription current seqnum [%d] is lesser than the event start seqnum [%d]",
                  currentSeqnum, event.getStartSeqnum()));
          List<ResourceMonitorEvent> priorEvents =
              service.getEvents(this.topic, currentSeqnum, event.getStartSeqnum());

          // Adiciona os eventos a partir do seqnum corrente até o primeiro seqnum do evento
          // notificado pelo monitor
          events.addAll(priorEvents);

          // Adiciona o evento notificado pelo monitor
          events.add(event);
        } else if (currentSeqnum > event.getStartSeqnum()
            && currentSeqnum <= event.getEndSeqnum()) {
          logger.debug(
              String.format(
                  "The subscription current seqnum [%d] is between the events start and end seqnum ]%d, %d]",
                  currentSeqnum, event.getStartSeqnum(), event.getEndSeqnum()));
          List<ResourceMonitorEvent> subsequentEvents =
              service.getEvents(this.topic, currentSeqnum, event.getEndSeqnum());

          // Adiciona os eventos a partir do seqnum corrente até último seqnum do evento notificado
          // pelo monitor
          events.addAll(subsequentEvents);
        } else if (currentSeqnum > event.getEndSeqnum()) {
          logger.debug(
              String.format(
                  "The subscription current seqnum [%d] is greater than the event end seqnum [%d]",
                  currentSeqnum, event.getEndSeqnum()));
          // Nada a fazer, pois o evento notificado pelo monitor é anteriores ao seqnum corrente
        } else {
          logger.debug(
              String.format(
                  "The subscription current seqnum [%d] is equal to the event start seqnum [%d]",
                  currentSeqnum, event.getStartSeqnum()));
          // Adiciona o evento notificado pelo monitor
          events.add(event);
        }

        for (ResourceMonitorEvent e : events) {
          Notification notification =
              new Notification(subscriptionId, topic.getTopicType(), mapper.map(e, topic));
          session.sendMessage(new TextMessage(objectMapper.writeValueAsBytes(notification)));
          this.currentSeqnum = event.getEndSeqnum();
        }
      } else {
        subscription.cancel();
        return;
      }
    } catch (IOException e) {
      logger.error("There was an IOException when trying to sendMessage", e);
      subscription.cancel();
      return;
    }
    subscription.request(1);
  }

  @Override
  synchronized public void onError(Throwable throwable) {
    logger.error(
        String.format(
            "Session [%s]: onError activated for topic [%s] with subscriptionId [%s]",
            sessionId, topic, subscriptionId),
        throwable);
  }

  @Override
  synchronized public void onComplete() {
    logger.debug(
        String.format(
            "Session [%s]: onComplete activated for topic [%s] with subscriptionId [%s]",
            sessionId, topic, subscriptionId));

    // Marca a flag indicativa de que não há mais eventos a receber
    this.completed = true;

    // Caso o onComplete seja chamada ante do onSubscribe o subscription estará nulo, então é necessário verificar
    if (subscription != null) {
      subscription.cancel();
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    TopicSubscriber that = (TopicSubscriber) o;
    return sessionId.equals(that.sessionId) && topic.equals(that.topic);
  }

  @Override
  public int hashCode() {
    return Objects.hash(sessionId, topic.getUUID());
  }
}
