package br.pucrio.tecgraf.soma.logsmonitor.manager;

import br.pucrio.tecgraf.soma.logsmonitor.flow.TopicPublisher;
import br.pucrio.tecgraf.soma.logsmonitor.flow.TopicSubscriber;
import br.pucrio.tecgraf.soma.logsmonitor.model.Topic;
import br.pucrio.tecgraf.soma.logsmonitor.model.mapper.TopicErrorsMapperFactory;
import br.pucrio.tecgraf.soma.logsmonitor.model.mapper.TopicEventMapper;
import br.pucrio.tecgraf.soma.logsmonitor.model.mapper.TopicEventMapperFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Lookup;
import org.springframework.stereotype.Component;

import java.util.*;

@Component
public class PublisherManager {
  private static final Log logger = LogFactory.getLog(PublisherManager.class);

  private final Map<Integer, TopicPublisher> publishersByTopic = new HashMap<>();
  private final Map<String, Set<TopicPublisher>> publishersBySession = new HashMap<>();

  @Autowired WebSocketSessionManager webSocketSessionManager;
  @Autowired TopicEventMapperFactory topicEventMapperFactory;
  @Autowired TopicErrorsMapperFactory topicErrorsMapperFactory;
  @Autowired ObjectMapper objectMapper;

  public synchronized TopicPublisher subscribe(
      Topic topic, Long initialSeqnum, String sessionId, String subscriptionId)
      throws IllegalArgumentException {
    logger.debug(String.format("Topic [%s]: add subscription for session [%s]", topic, sessionId));

    TopicPublisher publisher = getOrAddPublisherToTopicMap(topic);

    if (!publisher.isSubscribed(sessionId)) {
      TopicEventMapper mapper =
          topicEventMapperFactory.getEventMapperByTopicType(topic.getTopicType());
      TopicSubscriber subscriber =
          getTopicSubscriber(
              sessionId, subscriptionId, publisher.getTopic(), mapper, initialSeqnum);
      publisher.subscribe(sessionId, subscriber);
      addPublisherToSessionMap(sessionId, publisher);
    } else {
      String errorMsg =
          String.format(
              "Topic [%s]: subscription for session [%s] already exists",
              publisher.getTopic(), sessionId);
      logger.debug(errorMsg);
      throw new IllegalArgumentException(errorMsg);
    }

    return publisher;
  }

  private TopicPublisher getOrAddPublisherToTopicMap(Topic topic) {
    Integer topicUUID = topic.getUUID();
    TopicPublisher publisher;
    if (publishersByTopic.containsKey(topicUUID)) {
      publisher = publishersByTopic.get(topicUUID);
    } else {
      publisher = getTopicPublisher(topic);
    }
    logger.debug(String.format("Using publisher fot topic %s ", topic.getUUID()));
    publishersByTopic.putIfAbsent(topicUUID, publisher);
    return publisher;
  }

  private void addPublisherToSessionMap(String sessionId, TopicPublisher publisher) {
    Set<TopicPublisher> sessionPublishers =
        publishersBySession.getOrDefault(sessionId, new HashSet<>());
    sessionPublishers.add(publisher);
    publishersBySession.putIfAbsent(sessionId, sessionPublishers);
  }

  public synchronized Optional<TopicPublisher> unsubscribe(Topic topic, String sessionId) {
    logger.debug(
        String.format("Topic [%s]: remove subscription from session [%s] ", topic, sessionId));
    TopicPublisher publisher = null;
    if (publishersByTopic.containsKey(topic.getUUID())) {
      publisher = publishersByTopic.get(topic.getUUID());
      publisher.unsubscribe(sessionId);
      removePublisherFromSessionMap(sessionId, publisher);
      removePublisherFromTopicMap(topic, publisher);
    } else {
      logger.debug(String.format("Topic [%s]: does not exist", topic));
      // TODO: error.
    }

    return Optional.ofNullable(publisher);
  }

  private void removePublisherFromSessionMap(String sessionId, TopicPublisher publisher) {
    if (publishersBySession.containsKey(sessionId)) {
      Set<TopicPublisher> pubSet = publishersBySession.get(sessionId);
      pubSet.remove(publisher);
      if (pubSet.isEmpty()) {
        logger.debug(
            String.format("Session [%s]: remove session with zero subscriptions", sessionId));
        publishersBySession.remove(sessionId);
      }
    }
  }

  private void removePublisherFromTopicMap(Topic topic, TopicPublisher publisher) {
    if (!publisher.hasSubscribers()) {
      logger.debug(String.format("Topic [%s]: remove publisher with zero subscribers", topic));
      publishersByTopic.remove(topic.getUUID());
    }
  }

  public synchronized void onSessionClosed(String sessionId) {
    logger.debug(String.format("Session [%s]: remove all subscription", sessionId));
    if (publishersBySession.containsKey(sessionId)) {
      for (TopicPublisher publisher : publishersBySession.get(sessionId)) {
        publisher.unsubscribe(sessionId);
        removePublisherFromTopicMap(publisher.getTopic(), publisher);
      }
      publishersBySession.remove(sessionId);
    }
  }

  /*
  A anotação @Lookup usada nos métodos abaixo fazem com que o Spring substitua, em tempo de execução, a implementação
  stub de PublisherManager#getTopicPublisher e PublisherManager#getTopicSubscriber pelos construtores de TopicPublisher
  e TopicSubscriber, respectivamente. Observe que a assinatura do método precisa ser a mesma do construtor.

  Isso é usado para que a cada PublisherManager#subscribe uma nova instância de TopicPublisher e TopicSubscriber seja
  obtida, já que se a injeção de dependência fosse feita na instanciação de PublisherManager as mesmas instâncias de
  TopicPublisher e TopicSubscriber seriam usadas.

  Mais detalhes em:

  https://docs.spring.io/spring-framework/docs/5.3.3/reference/html/core.html#beans-factory-scopes-sing-prot-interaction
  https://docs.spring.io/spring-framework/docs/5.3.3/reference/html/core.html#beans-factory-lookup-method-injection
  */

  /*
  Repare que o uso da anotação @Lookup tem uma restrição:

  Such lookup methods can have default (stub) implementations that will simply
  get replaced by the container, or they can be declared as abstract - for the
  container to fill them in at runtime. In both cases, the container will
  generate runtime subclasses of the method's containing class via CGLIB, which
  is why such lookup methods can only work on beans that the container
  instantiates through regular constructors: i.e. lookup methods cannot get
  replaced on beans returned from factory methods where we cannot dynamically
  provide a subclass for them.

  Mais detalhes em:

  https://docs.spring.io/spring-framework/docs/5.3.3/javadoc-api/org/springframework/beans/factory/annotation/Lookup.html

   */

  /*
  Stub implementation which will be replaced by the container
  */
  @Lookup
  public TopicPublisher getTopicPublisher(Topic topic) {
    return null;
  }

  /*
  Stub implementation which will be replaced by the container
   */
  @Lookup
  public TopicSubscriber getTopicSubscriber(
      String sessionId,
      String subscriptionId,
      Topic topic,
      TopicEventMapper mapper,
      Long initialSeqnum) {
    return null;
  }
}
