package br.pucrio.tecgraf.soma.logsmonitor.model.validator;

import br.pucrio.tecgraf.soma.logsmonitor.model.Command;
import br.pucrio.tecgraf.soma.logsmonitor.model.JobLogsTopic;
import br.pucrio.tecgraf.soma.logsmonitor.model.Message;
import br.pucrio.tecgraf.soma.logsmonitor.model.Topic;
import br.pucrio.tecgraf.soma.logsmonitor.utils.ConstantsUtils;
import org.springframework.stereotype.Component;
import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;
import org.springframework.validation.Validator;

@Component
public class MessageValidator implements Validator {

  private final Validator topicValidator;

  public MessageValidator(Validator topicValidator) {
    if (topicValidator == null) {
      throw new IllegalArgumentException(
          "The supplied [Validator] is " + "required and must not be null.");
    }
    if (!topicValidator.supports(JobLogsTopic.class)) {
      throw new IllegalArgumentException(
          "The supplied [Validator] must " + "support the validation of [JobLogsTopic] instances.");
    }
    this.topicValidator = topicValidator;
  }

  @Override
  public boolean supports(Class<?> aClass) {
    return Message.class.isAssignableFrom(aClass);
  }

  @Override
  public void validate(Object o, Errors errors) {
    Message msg = (Message) o;

    validateSubscriptionId(msg.getSubscriptionId(), errors);
    validateSeqNum(msg.getCommand(), msg.getSeqnum(), errors);
    validateTopic(msg.getTopic(), errors);
  }

  private void validateSubscriptionId(String subsId, Errors errors) {
    Object[] args = {Message.JSON_PROPERTY_SUBSCRIPTION_ID};
    ValidationUtils.rejectIfEmptyOrWhitespace(
        errors,
        Message.JSON_PROPERTY_SUBSCRIPTION_ID,
        ConstantsUtils.VALIDATOR_INVALID_EMPTY_VALUE_ERROR_CODE,
        args,
        ConstantsUtils.VALIDATOR_INVALID_EMPTY_VALUE_ERROR_MSG);
  }

  private void validateTopic(Topic topic, Errors errors) {
    try {
      errors.pushNestedPath(Message.JSON_PROPERTY_TOPIC);
      ValidationUtils.invokeValidator(topicValidator, topic, errors);
    } finally {
      errors.popNestedPath();
    }
  }

  private void validateSeqNum(Command command, Long seqNum, Errors errors) {
    if (Command.SUBSCRIBE.equals(command)) {
      if (seqNum == null) {
        Object[] args = {Message.JSON_PROPERTY_SEQNUM};
        errors.rejectValue(
            Message.JSON_PROPERTY_SEQNUM,
            ConstantsUtils.VALIDATOR_MISSING_VALUE_ERROR_CODE,
            args,
            ConstantsUtils.VALIDATOR_MISSING_VALUE_ERROR_MSG);
      } else {
        if (seqNum.longValue() < 0) {
          Object[] args = {Message.JSON_PROPERTY_SEQNUM};
          errors.rejectValue(
              Message.JSON_PROPERTY_SEQNUM,
              ConstantsUtils.VALIDATOR_INVALID_NEGATIVE_VALUE_ERROR_CODE,
              args,
              ConstantsUtils.VALIDATOR_INVALID_NEGATIVE_VALUE_ERROR_MSG);
        }
      }
    }
  }
}
