package br.pucrio.tecgraf.soma.logsmonitor.websocket;

import br.pucrio.tecgraf.soma.logsmonitor.manager.WebSocketSessionManager;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.*;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;

import java.io.IOException;
import java.util.Map;

@Service
public class WebSocketNotificatioErrorService {
  private static final Log logger = LogFactory.getLog(WebSocketNotificatioErrorService.class);
  @Autowired ObjectMapper objectMapper;
  @Autowired private WebSocketSessionManager webSocketSessionManager;

  public void onErrorNotify(
      String sessionId,
      String subscriptionId,
      ErrorType errorType,
      String errorMsg,
      Map<String, Object> args) {
    String errorSubsId = subscriptionId == null ? "" : subscriptionId;
    ErrorEvent errorEvent = getErrorEvent(errorType, args);
    NotificationError notificationError =
        new NotificationError(errorSubsId, errorType, errorMsg, errorEvent);
    final WebSocketSession session = webSocketSessionManager.getSession(sessionId);
    if (session != null) {
      try {
        if (session.isOpen()) {
          session.sendMessage(new TextMessage(objectMapper.writeValueAsBytes(notificationError)));
        }
      } catch (IOException e) {
        logger.error("Cannot notify error message to websocket channel", e);
      }
    }
  }

  private ErrorEvent getErrorEvent(ErrorType errorType, Map<String, Object> args) {
    ErrorEvent errorEvent = null;
    switch (errorType) {
      case INVALID_MESSAGE:
        String exceptionMsg =
            (String) args.get(InvalidMessageErrorEvent.JSON_PROPERTY_EXCEPTION_MESSAGE);
        errorEvent = new InvalidMessageErrorEvent(exceptionMsg);
        break;
      case PROJECT_PERMISSION_ERROR:
        String projectId = (String) args.get(ProjectPermissionErrorEvent.JSON_PROPERTY_PROJECT_ID);
        errorEvent = new ProjectPermissionErrorEvent(projectId);
        break;
      case MISSING_FIELD_ERROR:
        String fieldName = (String) args.get(MissingFieldErrorEvent.JSON_PROPERTY_FIELD_NAME);
        errorEvent = new MissingFieldErrorEvent(fieldName);
        break;
      case INVALID_FIELD_VALUE_ERROR:
        String invalidFieldName =
            (String) args.get(InvalidFieldValueErrorEvent.JSON_PROPERTY_FIELD_NAME);
        String invalidFieldValue =
            String.valueOf(args.get(InvalidFieldValueErrorEvent.JSON_PROPERTY_FIELD_VALUE));
        errorEvent = new InvalidFieldValueErrorEvent(invalidFieldName, invalidFieldValue);
        break;
      case RESOURCE_ERROR:
        String details = (String) args.get(ResourceErrorEvent.JSON_PROPERTY_DETAILS);
        errorEvent = new ResourceErrorEvent(details);
        break;
      default:
        logger.error("Unknown error type " + errorType);
    }
    return errorEvent;
  }
}
