package br.pucrio.tecgraf.soma.job.log.watcher.impl;

import br.pucrio.tecgraf.soma.job.log.watcher.event.FileEvent;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatchEventListener;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatcher;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import static java.lang.String.format;
public class PollingFileWatcher implements IFileWatcher {

  private final Logger LOG = LoggerFactory.getLogger(PollingFileWatcher.class);
  private final List<IFileWatchEventListener> listeners;
  /** O diretório monitorado */
  private Path watchedDirectoryPath;
  private FileAlterationObserver observer;
  private FileAlterationMonitor monitor;

  private volatile boolean isWatching = false;

  public PollingFileWatcher(Integer pollingIntervalMillis) throws IOException {
    this(new CopyOnWriteArrayList<>(), pollingIntervalMillis);
  }

  /**
   * Construtor para os testes.
   *
   * @param listeners lista de ouvintes
   * @param pollingIntervalMillis intervalo de polling em milissegundos
   */
  public PollingFileWatcher(List<IFileWatchEventListener> listeners, Integer pollingIntervalMillis) {
    this.monitor = new FileAlterationMonitor(pollingIntervalMillis);
    this.listeners = listeners;
  }

  @Override
  public void register(final String dirStrPath, FileFilter fileFilter) throws IOException {

    LOG.debug("[Polling Watcher] register [{}]", dirStrPath);
    watchedDirectoryPath = Paths.get(dirStrPath);
    if (!Files.isDirectory(watchedDirectoryPath)) {
      final String msg = format("The path %s is not a directory!", dirStrPath);
      LOG.error(msg);
      throw new IOException(msg);
    }
    LOG.info(
        "Registering Polling Watcher for directory {} [Real path={}]",
        dirStrPath,
        getRealPath(watchedDirectoryPath));
    this.observer = new FileAlterationObserver(dirStrPath, fileFilter);
  }

  @Override
  public void startWatch() throws InterruptedException {
    LOG.info("Polling Watcher Service starting...");
    try {
      observer.addListener(new FileAlterationListenerImplementation());
      monitor.addObserver(observer);
      monitor.start();
      isWatching = true;
    } catch (Exception e) {
      LOG.debug("Polling Watcher Service could not be started...", e);
    }
  }

  @Override
  public void stopWatch() throws Exception {
    LOG.info("Polling Watcher Service stopping...");
    this.monitor.stop();
    this.monitor.removeObserver(observer);
    isWatching = false;
  }

  public boolean isWatching() {
    return isWatching;
  }

  @Override
  public void addFileWatchEventListener(IFileWatchEventListener listener) {
    this.listeners.add(listener);
  }

  @Override
  public void removeFileWatchEventListener(IFileWatchEventListener listener) {
    this.listeners.remove(listener);
  }

  @Override
  public void close() throws IOException {
    LOG.debug("Polling Watcher was closed...");
    try {
      stopWatch();
    } catch (Exception e) {
      throw new IOException("Polling Watcher error on closing: ", e);
    }
  }

  protected synchronized void processEvent(File file) {
    FileEvent fe = new FileEvent(file);
    LOG.info(
        "Polling Watcher Service notifying change event for file {} [Real path={}]",
        file,
        getRealPath(file.toPath()));
    for (IFileWatchEventListener listener : this.listeners) {
      listener.onFileModified(fe);
    }
  }

  private final class FileAlterationListenerImplementation extends FileAlterationListenerAdaptor {
    @Override
    public void onFileChange(File file) {
      // Só é necessário ouvir os eventos de modificação dos arquivos (eventos de criação/remoção são ignorados)
      processEvent(file);
    }
  }

  private Path getRealPath(Path path) {
    Path realPath = null;
    try {
      realPath = path.toRealPath();
    } catch (IOException e) {
      LOG.error("Erro while getting real path for {}", path);
    }

    return realPath;
  }
}
