package br.pucrio.tecgraf.soma.logsmonitor.appservice;

import br.pucrio.tecgraf.soma.logsmonitor.model.Message;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ErrorType;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.InvalidFieldValueErrorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.model.validator.MessageValidator;
import br.pucrio.tecgraf.soma.logsmonitor.service.TopicService;
import br.pucrio.tecgraf.soma.logsmonitor.service.TopicServiceFactory;
import br.pucrio.tecgraf.soma.logsmonitor.websocket.WebSocketNotificatioErrorService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.socket.WebSocketSession;

import java.util.HashMap;
import java.util.Map;

@Service
public class JobLogsAppService {
  private static final Log logger = LogFactory.getLog(JobLogsAppService.class);
  @Autowired private TopicServiceFactory serviceFactory;

  @Autowired private MessageValidator messageValidator;

  @Autowired private WebSocketNotificatioErrorService webSocketErrorService;

  public void processMessage(Message message, WebSocketSession session) {
    TopicService service = serviceFactory.getServiceByTopicType(message.getTopic().getTopicType());

    if (messageHasErrors(message, session.getId())) {
      return;
    }

    switch (message.getCommand()) {
      case SUBSCRIBE:
        service.subscribe(
            message.getTopic(), session, message.getSubscriptionId(), message.getSeqnum());
        break;
      case UNSUBSCRIBE:
        service.unsubscribe(session.getId(), message.getTopic());
        break;
    }
  }

  private boolean messageHasErrors(Message message, String sessionId) {
    BeanPropertyBindingResult result = new BeanPropertyBindingResult(message, "message");
    messageValidator.validate(message, result);

    if (result.hasErrors()) {
      FieldError fieldError = result.getFieldErrors().get(0);
      notifyValidationErrors(message.getSubscriptionId(), sessionId, fieldError);
      return true;
    }
    return false;
  }

  private void notifyValidationErrors(
      String subscriptionId, String sessionId, FieldError fieldError) {
    String errorMsg = String.format(fieldError.getDefaultMessage(), fieldError.getField());
    logger.error(errorMsg);
    Map<String, Object> map = new HashMap<>();
    map.put(InvalidFieldValueErrorEvent.JSON_PROPERTY_FIELD_NAME, fieldError.getField());
    map.put(InvalidFieldValueErrorEvent.JSON_PROPERTY_FIELD_VALUE, fieldError.getRejectedValue());
    webSocketErrorService.onErrorNotify(
        sessionId, subscriptionId, ErrorType.INVALID_FIELD_VALUE_ERROR, errorMsg, map);
  }
}
