package br.pucrio.tecgraf.soma.logsmonitor.model.validator;

import br.pucrio.tecgraf.soma.logsmonitor.model.JobLogsTopic;
import br.pucrio.tecgraf.soma.logsmonitor.utils.ConstantsUtils;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;
import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;
import org.springframework.validation.Validator;

import java.util.Base64;

@Component
@Primary
public class JobLogsTopicValidator implements Validator {

  @Override
  public boolean supports(Class<?> aClass) {
    return JobLogsTopic.class.isAssignableFrom(aClass);
  }

  @Override
  public void validate(Object o, Errors errors) {
    JobLogsTopic jlt = (JobLogsTopic) o;
    validateAttrs(JobLogsTopic.JSON_PROPERTY_PROJECT_ID, jlt.getProjectId(), errors, jlt);
    validateAttrs(JobLogsTopic.JSON_PROPERTY_JOB_ID, jlt.getJobId(), errors, jlt);

    validateFlowNodeId(jlt.getFlowNodeId(), errors);
    validateIfEmpty(JobLogsTopic.JSON_PROPERTY_LOG_NAME, jlt.getLogName(), errors);
  }

  private void validateAttrs(String attrName, String attrValue, Errors errors, JobLogsTopic jlt) {
    validateIfEmpty(attrName, attrValue, errors);
    validateNotBase64Encoded(attrName, attrValue, errors, jlt);
  }

  private void validateIfEmpty(String fieldName, String fieldValue, Errors errors) {
    if (fieldValue != null) {
      Object[] args = {fieldName};
      ValidationUtils.rejectIfEmptyOrWhitespace(
          errors,
          fieldName,
          ConstantsUtils.VALIDATOR_INVALID_EMPTY_VALUE_ERROR_CODE,
          args,
          ConstantsUtils.VALIDATOR_INVALID_EMPTY_VALUE_ERROR_MSG);
    }
  }

  private void validateNotBase64Encoded(
      String fieldName, String fieldValue, Errors errors, JobLogsTopic jlt) {
    Object[] args = {fieldName};
    try {
      Base64.getDecoder().decode(fieldValue);
    } catch (IllegalArgumentException e) {
      errors.rejectValue(
          fieldName,
          ConstantsUtils.VALIDATOR_NOT_BASE64_ENCODED_ERROR_CODE,
          args,
          ConstantsUtils.VALIDATOR_NOT_BASE64_ENCODED_ERROR_MSG);
    }
  }

  private void validateFlowNodeId(Integer flowNodeId, Errors errors) {
    if (flowNodeId != null) {
      if (flowNodeId < 0) {
        Object[] args = {JobLogsTopic.JSON_PROPERTY_FLOW_NODE_ID};
        errors.rejectValue(
            JobLogsTopic.JSON_PROPERTY_FLOW_NODE_ID,
            ConstantsUtils.VALIDATOR_INVALID_NEGATIVE_VALUE_ERROR_CODE,
            args,
            ConstantsUtils.VALIDATOR_INVALID_NEGATIVE_VALUE_ERROR_MSG);
      }
    }
  }
}
