package br.pucrio.tecgraf.soma.logsmonitor.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientRequestException;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import reactor.core.publisher.Mono;

import java.time.Duration;
import java.util.List;

@Service
public class ProjectService {
  private final Logger logger = LoggerFactory.getLogger(ProjectService.class);

  @Value("${job.logs.monitor.services.projectService.projectPermission.path}")
  private String projectsPermissionPath;

  @Value("${job.logs.monitor.services.projectService.projectPermission.enable}")
  private Boolean projectsPermissionCheck;

  private WebClient webClient;

  public ProjectService(
      @Value("${job.logs.monitor.services.projectService.clientBaseUrl}") String clientBaseUrl) {
    this.webClient =
        WebClient.builder()
            .baseUrl(clientBaseUrl)
            .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
            .build();
  }

  public List<String> getUserProjects(String accessToken) throws WebClientResponseException {
    final Mono<List> response =
        webClient
            .get()
            .uri(projectsPermissionPath)
            .headers(headers -> headers.setBearerAuth(accessToken))
            .retrieve()
            .bodyToMono(new ParameterizedTypeReference<List>() {})
            .timeout(Duration.ofMillis(60000));

    final List<String> projects = response.block();
    return projects;
  }

  public boolean hasUserProjectPermission(String accessToken, String projectId) {
    try {
      if(!projectsPermissionCheck){
        logger.info("Skipping project permission check");
        return true;
      }

      logger.info(String.format("Checking user permission to project %s", projectId));
      final Mono<Boolean> response =
          webClient
              .get()
              .uri(projectsPermissionPath)
              .headers(headers -> headers.setBearerAuth(accessToken))
              .retrieve()
              .bodyToMono(new ParameterizedTypeReference<List<String>>() {})
              .flatMap(
                  list -> {
                    for (String p : list) {
                      if (p.equals(projectId)) {
                        return Mono.just(true);
                      }
                    }
                    return Mono.just(false);
                  });
      return response.block();
    } catch (WebClientRequestException e) {
      throw new ServiceException(e.getMessage());
    } catch (WebClientResponseException ex) {
      throw new ServiceException(ex.getMessage(), ex.getRawStatusCode());
    }
  }
}
