/**
 *  JavaScript formatters and mask controls.
 *
 *  Leonardo Quijano
 *  Feb 2006
 *
 **/

dojo.require("dojo.lang.*");
dojo.require("dojo.event.*");

/** ********************* METHODS ********************* */
/**
 * Connects a formatter for the specified field. This binds several
 * events to format and mask the current value. The formatter should be
 * an object that specify the function contract for formatting strings
 * (see the StubFormatter object below).
 * @param field the field to apply formatting rules.
 * @param formatter the formatting rules.
 */
function connectFormatter(field, formatter) {
    dojo.debug("Connecting formatter for field: " + field.id);

    field._formatter = formatter;

    // Format on init.
    var str = formatter.format(field.value);
    if(str) {
        field.value = str;
    }

    dojo.event.kwConnect({
        srcObj: field,
        srcFunc: "onchange",
        targetFunc: "handleFormatEvent",
        once: true
    });
    dojo.event.kwConnect({
        srcObj: field,
        srcFunc: "onkeypress",
        targetFunc: "handleKeyPressEvent",
        once: true
    });
}

/**
 * Disconnect the formatter events for the specified field.
 * @param field the field.
 */
function disconnectFormatter(field) {
    dojo.debug("Disconnecting formatter for field: " + field.id);

    dojo.event.kwDisconnect({
        srcObj: field,
        srcFunc: "onchange",
        targetFunc: "handleFormatEvent",
        once: true
    });
    dojo.event.kwDisconnect({
        srcObj: field,
        srcFunc: "onkeypress",
        targetFunc: "handleKeyPressEvent",
        once: true
    });
}

/**
 * Handles a formatting event (usually used with the onchange
 * event on a textfield).
 * @param event the event.
 */
function handleFormatEvent(event) {
    if(event && event.target) {
        var field = event.target;
        var value = field.value;
        dojo.debug("Formatting field: " + field.id);

        var formatter = field._formatter;
        if(!formatter) {
            dojo.debug("No formatter configured for field: " + field.id);
        }
        else {
            var str = formatter.format(value);
            if(str) {
                field.value = str;
            }
            else {
                dojo.debug("Formatting failed for value: " + value);
            }
        }
    }
}

/**
 * Handles a keypress event (used with input masks).
 */
function handleKeyPressEvent(event) {
    var field = event.target;
    var code = event.charCode;
    var formatter = field._formatter;
    var value = field.value;
    var selectionStart = getSelectionStart(field);
    var selectionEnd = getSelectionEnd(field);
    // Discard the currently selected text (which will be replaced)
    // and check if the code is allowed in the selection.
    var previousStr = value.substr(0, selectionStart);
    var nextStr = value.substr(selectionEnd, value.length - selectionEnd);
    var allowed = true;

    if(isSpecialKey(code)) {
        allowed = formatter.isAllowed(code, previousStr, nextStr);
    }
    else {
        var ch = String.fromCharCode(code);

        // We'll handle the value manually from now on.
        field.value = previousStr;

        // Check if the formatter can help us with some chars.
        if(formatter.nextFixedChars) {
            var fixedChars = formatter.nextFixedChars(code, previousStr, nextStr);
            if(fixedChars != "") {
                previousStr = previousStr + fixedChars;
                field.value = previousStr;

                // Ignore ch if the fixed chars start with it.
                if(dojo.string.startsWith(fixedChars, ch)) {
                    allowed = false;
                }
            }
        }

        // Check if the code can be inserted now.
        if(allowed) {
            allowed = formatter.isAllowed(code, previousStr, nextStr);
        }

        // If the formatter wants to assemble, let him.
        if(formatter.assembleValue) {
            field.value = formatter.assembleValue(
                previousStr, (allowed ? ch : ""), nextStr);
        }
        else {
            field.value = previousStr + (allowed ? ch : "") + nextStr;
        }

        if(allowed) {
            setCaretToPos(field, previousStr.length + 1);
        }
        else {
            setCaretToPos(field, previousStr.length);
        }

        // Override standard handling.
        event.preventDefault();
    }
}

/** ********************* FORMATTERS ********************* */
/**
 * Formatters should include the following methods:
 * - format(str):
 *       Returns the specified spring, formatted using the formatter's rules.
 * - validate(str):
 *       Validates the string, returning whether it is a match,
 *       partial match, or no match at all.
 * - isAllowed(code, previousStr, nextStr, keys): Validates whether the specified
 *       code (a character or a keyboard code) is permitted to be
 *       inserted between previousStr and nextStr.
 */

/** Stub formatter. Does nothing. */
function StubFormatter() {
    this.format = function(str) { return str; };
    this.validate = function(str) { return true; };
    this.isAllowed = function(code, previousStr, nextStr) { return true; };
}
