package org.tecgraf.tdk.cache;

import org.geotools.data.Query;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.Feature;
import org.opengis.feature.type.FeatureType;

/**
 * This Interface defines the classes responsible for maintaining 
 * the feature cache itself. It intends to be spatially indexed . It does not have access to the features
 * source.  
 * @author fmoura
 * @since TDK 3.0.0
 */
public interface FeatureCacher<T extends FeatureType, F extends Feature >
{
	/**
	 * Adds new features to the cache. It combines the given {@link Query} with 
	 * the previous stored query to insure the query represents the given collection. 
	 * @param featureCollection The collection of features to be added to the cache
	 * @param query The {@link Query} that represents the given collection
	 */
	void add(FeatureCollection<T,F> featureCollection, Query query);
	
	/**
	 * Adds a new feature to the cache. It verifies if the given feature is accepted
	 * by the previous stored query. if not, it modifies the query so it represents
	 * the entire feature collection stored in the cacher.
	 * @param feature
	 */
	void add(F feature);
	
	/**
	 * Updates this feature property values at the cache.
	 * @param feature The feature to be updated at the memory cache.
	 */
	void update(F feature);
	
	/**
	 * Removes the given feature from the cache.
	 * @param feature
	 */
	void remove(F feature);
	
	/**
	 * Clear all the cache.
	 */
	void clear();
	
	/**
	 * Retrieves the cached features that match the given query. Note that the cacher should not
	 * have access to the feature source, so this method will not get all the possible features,
	 * only the cached ones.
	 * @param query The query to be used as filter to retrieving the features
	 * @return A {@link FeatureCollection} that stores the cached features that matches the given query. 
	 */
	FeatureCollection<T,F> get(Query query);
	
	/**
	 * Returns the {@link Query} that matches all the stored features.
	 * @return the {@link Query} that matches all the stored features. 
	 */
	Query getCachedQuery();

	/**
	 * 
	 * @return The name of the type this instance stores features for.
	 */
	String getTypeName();

	/**
	 * Gets the feature type of the data stored at the cache.
	 * @return The feature type of the data stored at the cache
	 */
	T getFeatureType();
}
