/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: CachingDataStore.java
 * created: May 21, 2009
 */
package org.tecgraf.tdk.cache;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import org.apache.log4j.Logger;
import org.geotools.data.DataStore;
import org.geotools.data.DefaultQuery;
import org.geotools.data.FeatureReader;
import org.geotools.data.FeatureSource;
import org.geotools.data.FeatureStore;
import org.geotools.data.FeatureWriter;
import org.geotools.data.LockingManager;
import org.geotools.data.Query;
import org.geotools.data.ServiceInfo;
import org.geotools.data.Transaction;
import org.geotools.feature.SchemaException;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.Name;
import org.opengis.filter.Filter;
import org.tecgraf.tdk.cache.query.DefaultQueryAnalyzer;
import org.tecgraf.tdk.cache.query.QueryAnalyzer;

/**
 * This class is a DataStore wrapper that manages a feature cache upon all the feature access classes
 * like {@link FeatureReader}, {@link FeatureWriter}, {@link FeatureSource} and {@link FeatureStore}.
 * @author fabiomano, fmoura
 */
public class SimpleCachingDataStore implements CachingDataStore
{
	private static Logger _logger = Logger.getLogger(SimpleCachingDataStore.class);
    DataStore _wrapped;
    
    private Map<String, FeatureCacher<SimpleFeatureType,SimpleFeature>> _featureCachers 
    = new WeakHashMap<String, FeatureCacher<SimpleFeatureType,SimpleFeature>>();
    
    CachingFeatureAccessFactory<SimpleFeatureType, SimpleFeature> _cachingFeatureAccessFactory = new DefaultCachingFeatureAccessFactory();
    
    public SimpleCachingDataStore(DataStore original)
    {
        if (original == null)
            throw new IllegalArgumentException("original can't be null");
        
        _wrapped = original;
    }
    
    /**
     * Sets a new {@link CachingFeatureAccessFactory} to be used by this {@link SimpleCachingDataStore}
     * to create the feature access instances of ( {@link FeatureSource}, {@link FeatureStore}, {@link FeatureReader} and {@link FeatureWriter} )
     * @param cachingFeatureAccessFactory The {@link CachingFeatureAccessFactory} to be set.
     */
    public void setCachingFeatureAccessFactory(CachingFeatureAccessFactory<SimpleFeatureType, SimpleFeature> cachingFeatureAccessFactory)
    {
    	_cachingFeatureAccessFactory = cachingFeatureAccessFactory;
    }
    

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getFeatureReader(org.geotools.data.Query, org.geotools.data.Transaction)
     */
    @Override
    public FeatureReader<SimpleFeatureType, SimpleFeature> getFeatureReader(Query query, Transaction transaction)
            throws IOException
    {
        String typeName = query.getTypeName();
        
        //builds a cacher if one does not exists.
        buildFeatureCacher(typeName);
        FeatureCacher<SimpleFeatureType,SimpleFeature> cacher = _featureCachers.get(typeName);
        
        //tests if the feature cacher has the necessary features stored (if the current query is the same or a sub-query of the query at the cacher)
        QueryAnalyzer queryAnalyzer = new DefaultQueryAnalyzer(cacher.getCachedQuery());
        
        if(!queryAnalyzer.isSubQuery(query))
        {
            FeatureSource<SimpleFeatureType, SimpleFeature> source = _wrapped.getFeatureSource(typeName);
            Query includeAllQuery = new DefaultQuery(query.getTypeName(),Filter.INCLUDE);
            cacher.add(source.getFeatures(includeAllQuery),includeAllQuery);
        }
        
        return _cachingFeatureAccessFactory.createFeatureReader(query, cacher);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getFeatureWriter(java.lang.String, org.geotools.data.Transaction)
     */
    @Override
    public FeatureWriter<SimpleFeatureType, SimpleFeature> getFeatureWriter(String typeName, Transaction transaction)
            throws IOException
    {
    	buildFeatureCacher(typeName);
        return buildCachingFeatureWriter(typeName,_wrapped.getFeatureWriter(typeName, transaction));
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getFeatureWriter(java.lang.String, org.opengis.filter.Filter, org.geotools.data.Transaction)
     */
    @Override
    public FeatureWriter<SimpleFeatureType, SimpleFeature> getFeatureWriter(String typeName, Filter filter,
            Transaction transaction) throws IOException
    {
    	buildFeatureCacher(typeName);
        return buildCachingFeatureWriter(typeName,_wrapped.getFeatureWriter(typeName, filter, transaction));
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getFeatureWriterAppend(java.lang.String, org.geotools.data.Transaction)
     */
    @Override
    public FeatureWriter<SimpleFeatureType, SimpleFeature> getFeatureWriterAppend(String typeName,
            Transaction transaction) throws IOException
    {
    	buildFeatureCacher(typeName);
        return buildCachingFeatureWriter(typeName,_wrapped.getFeatureWriterAppend(typeName, transaction));
    }
    
    /**
     * Creates a new CachingFeatureWriter wrapping the feature writer informed.
     * @param typeName 
     * @param wrappedWriter Feature Writer to wrap.
     * @return The CachingFeatureWriter created.
     * @throws IOException
     */
    private FeatureWriter<SimpleFeatureType, SimpleFeature> buildCachingFeatureWriter(String typeName, FeatureWriter<SimpleFeatureType, SimpleFeature> wrappedWriter) throws IOException
    {
//        buildFeatureCacher(typeName);
        FeatureCacher<SimpleFeatureType,SimpleFeature> cacher = _featureCachers.get(typeName);
        return _cachingFeatureAccessFactory.createFeatureWriter(wrappedWriter, cacher);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getLockingManager()
     */
    @Override
    public LockingManager getLockingManager()
    {
        return _wrapped.getLockingManager();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getSchema(java.lang.String)
     */
    @Override
    public SimpleFeatureType getSchema(String typeName) throws IOException
    {
        return _wrapped.getSchema(typeName);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getTypeNames()
     */
    @Override
    public String[] getTypeNames() throws IOException
    {
        return _wrapped.getTypeNames();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getView(org.geotools.data.Query)
     */
    @Override
    public FeatureSource<SimpleFeatureType, SimpleFeature> getView(Query query) throws IOException, SchemaException
    {
        return _wrapped.getView(query);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#updateSchema(java.lang.String, org.opengis.feature.simple.SimpleFeatureType)
     */
    @Override
    public void updateSchema(String typeName, SimpleFeatureType featureType) throws IOException
    {
        _wrapped.updateSchema(typeName, featureType);
        clearCache(typeName);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#createSchema(org.opengis.feature.type.FeatureType)
     */
    @Override
    public void createSchema(SimpleFeatureType featureType) throws IOException
    {
        _wrapped.createSchema(featureType);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#dispose()
     */
    @Override
    public void dispose()
    {
        _wrapped.dispose();
        clearCache();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#getInfo()
     */
    @Override
    public ServiceInfo getInfo()
    {
        return _wrapped.getInfo();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#getNames()
     */
    @Override
    public List<Name> getNames() throws IOException
    {
        return _wrapped.getNames();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#getSchema(org.opengis.feature.type.Name)
     */
    @Override
    public SimpleFeatureType getSchema(Name name) throws IOException
    {
        return _wrapped.getSchema(name);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#updateSchema(org.opengis.feature.type.Name, org.opengis.feature.type.FeatureType)
     */
    @Override
    public void updateSchema(Name typeName, SimpleFeatureType featureType) throws IOException
    {
        _wrapped.updateSchema(typeName, featureType);
        clearCache(typeName.getLocalPart());
    }

    /**
     * Returns the original datastore, wrapped inside the caching datastore.
     * @return
     */
    public DataStore getTargetDataStore()
    {
        return _wrapped;
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.DataStore#getFeatureSource(java.lang.String)
     */
    @Override
    public FeatureSource<SimpleFeatureType, SimpleFeature> getFeatureSource(String typeName) throws IOException
    {
        buildFeatureCacher(typeName);
        
        FeatureCacher<SimpleFeatureType,SimpleFeature> cacher = _featureCachers.get(typeName);
        
        FeatureSource<SimpleFeatureType, SimpleFeature> featureSource = _wrapped.getFeatureSource(typeName);
    	
        CachingFeatureSource<SimpleFeatureType,SimpleFeature> source = null; 
    	if(featureSource instanceof FeatureStore)
    	{
    		source = _cachingFeatureAccessFactory.createFeatureStore((FeatureStore<SimpleFeatureType, SimpleFeature>) featureSource, cacher);
    	}
    	else
    	{
    		source = _cachingFeatureAccessFactory.createFeatureSource(featureSource, cacher);
    	}
        
        return source;
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.DataAccess#getFeatureSource(org.opengis.feature.type.Name)
     */
    @Override
    public FeatureSource<SimpleFeatureType, SimpleFeature> getFeatureSource(Name typeName) throws IOException
    {
        return getFeatureSource(typeName.getLocalPart());
    }

	@Override
	public void clearCache() {
	    _featureCachers.clear();
//		for(FeatureCacher<SimpleFeatureType,SimpleFeature> featureCacher : _featureCachers.values())
//		{
//			featureCacher.clear();
//		}
		
	}

	@Override
	public void clearCache(String typename) {
		if (_featureCachers.containsKey(typename))
		{
			_featureCachers.remove(typename);
		}
		
	}
	
	/**
	 * Builds the cacher for the typeName informed if it doesn't exists. 
	 * @param typeName
	 * @throws IOException 
	 */
	private void buildFeatureCacher(String typeName) throws IOException
	{
        if (!_featureCachers.containsKey(typeName))
        {
            _logger.debug("Type '"+typeName+"' not cached, creating new cacher.");
            FeatureCacher<SimpleFeatureType, SimpleFeature> cacher  = _cachingFeatureAccessFactory.createFeatureCacher(getSchema(typeName));
            _featureCachers.put(typeName, cacher);
        }	    
	}

//	@Override
//	public void reloadCache() throws IOException {
//		for(FeatureCacher<SimpleFeatureType,SimpleFeature> featureCacher : _featureCachers.values())
//		{
//			reloadCacher(featureCacher);
//		}
//		
//	}
//
//	@Override
//	public void reloadCache(String typename) throws IOException {
//		if (_featureCachers.containsKey(typename))
//		{
//			FeatureCacher<SimpleFeatureType, SimpleFeature> featureCacher = _featureCachers.get(typename);
//			reloadCacher(featureCacher);
//		}
//	}
//	
//	private void reloadCacher(
//			FeatureCacher<SimpleFeatureType, SimpleFeature> featureCacher)
//			throws IOException {
//		featureCacher.clear();
//		Query query = featureCacher.getCachedQuery();
//		FeatureSource<SimpleFeatureType, SimpleFeature> featureSource = getFeatureSource(featureCacher.getTypeName());
//		featureSource.getFeatures(query);
//	}
    
}
