/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 * 
 *    (C) 2004-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.renderer.style;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;

import org.geotools.geometry.jts.GeomCollectionIterator;
import org.geotools.geometry.jts.LiteShape2;
import org.geotools.renderer.lite.GlyphRenderer;
import org.geotools.styling.ExternalGraphic;
import org.geotools.styling.Graphic;

import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryCollection;

/**
 * Represents a {@link Style2D} that can be rendered by a {@link GlyphRenderer}. 
 * 
 * @author milton
 */
public class GlyphStyle2D extends Style2D
{
	private static final AffineTransform IDENTITY_TRANSFORM = new AffineTransform();
	private GlyphRenderer _renderer ;
	private ExternalGraphic _externalGraphic;
	private Graphic _graphic;
	private Object _feature;
	private int _height;
	
	/**
	 * Constructor.
	 * 
	 * @param renderer GlyphRenderer to be used for rendering this GlyphStyle2D
	 * @param graphic Graphic for defining the glyph.
	 * @param externalGraphic ExternalGraphic for defining the glyph.
	 * @param feature Feature based on which the glyph will be rendered. 
	 * @param size Size of the glyph to be rendered.
	 */
	public GlyphStyle2D(GlyphRenderer renderer, Graphic graphic, ExternalGraphic externalGraphic, Object feature, int size)
	{
		_renderer = renderer;
		_graphic = graphic;
		_externalGraphic = externalGraphic;
		_feature = feature;
		_height = size;
	}
	
	/**
	 * Requests the glyph to be rendered on a specified Graphics. 
	 *  
	 * @param graphics the Graphics on which to render.
	 * @param shape a Shape whose coordinates will be used for rendering. 
	 * @param scale the scale of the map being rendered.
	 */
	public void render(Graphics graphics, Shape shape, double scale)
	{
		if (!isScaleInRange(scale)) return;
		
		Graphics2D g = (Graphics2D)graphics;
		
		// extracts a PathIterator from the shape
		PathIterator citer;
		if (shape instanceof LiteShape2)
		{
			// if shape is a LiteShape2, then the following procedure must be performed
			LiteShape2 liteShape = (LiteShape2)shape;
			GeometryCollection gc;
			if (liteShape.getGeometry() instanceof GeometryCollection)
				gc = (GeometryCollection) liteShape.getGeometry();
			else {
				Geometry[] gs = new Geometry[1];
				gs[0] = liteShape.getGeometry();
				gc = liteShape.getGeometry().getFactory().createGeometryCollection(
						gs); // make a Point,Line, or Poly into a GC
			}
			citer = new GeomCollectionIterator(gc,
					IDENTITY_TRANSFORM, false, 1.0);
		}
		else
		{
			// if shape is NOT a LiteShape2, then Shape.getPathIterator() is fine
			citer = shape.getPathIterator(IDENTITY_TRANSFORM);
		}

		// get the point onto the shape has to be painted
		float[] coords = new float[2];
		while (!(citer.isDone()))
		{
			citer.currentSegment(coords);
			_renderer.render(g, _graphic, _externalGraphic, _feature, _height,coords[0],coords[1]);
			citer.next();
		}		
	}
}
