package org.geotools.data.terralib;

import java.io.IOException;
import java.util.Timer;
import java.util.UUID;
import java.util.concurrent.Semaphore;

import org.apache.log4j.Logger;
import org.geotools.data.terralib.exception.IllegalStateException;
import org.geotools.data.terralib.exception.InvalidCrsWktException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.swig.DBConnection;
import org.geotools.data.terralib.swig.PersistenceTransferenceObject;
import org.geotools.data.terralib.swig.StringVector;
import org.geotools.data.terralib.swig.TeAttrDataType;
import org.geotools.data.terralib.swig.TeGeomRep;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptor;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptorVector;
import org.geotools.data.terralib.swig.TerralibServiceNative;
import org.geotools.data.terralib.util.thread.Watchdog;

import com.vividsolutions.jts.geom.Envelope;


public class ThreadBlockingTerralibService extends TerralibServiceNative {

	private static final StackTraceElement[] EMPTY_STACK = new StackTraceElement[0];
	private static final long NO_THREAD = -1;

	private static final int WATCHDOG_TIMEOUT_MILLISECONDS = 60000;// One minute.

	private static final int WATCHDOG_CHECK_STEP_MILLISECONDS = 100;

	Logger _logger = Logger.getLogger(ThreadBlockingTerralibService.class);
	
	private Semaphore _semaphore = new Semaphore(1);
	private boolean _enableLock;
	private Watchdog _watchdog = new Watchdog(WATCHDOG_CHECK_STEP_MILLISECONDS);

	private StackTraceElement[] _lastSuccessfullCallerStack = EMPTY_STACK;
	private long _lastSuccessfullThreadID = NO_THREAD;

	protected ThreadBlockingTerralibService(DBConnection connection, boolean enableLock) {
		super(connection);
		
		_enableLock = enableLock;
		
		if(_enableLock)
			_logger.info("Using locking mechanism.");
		else
			_logger.info("NOT using locking mechanism.");
		
		Timer timer = new Timer();
		timer.schedule(_watchdog, 0, WATCHDOG_CHECK_STEP_MILLISECONDS);
		
	}

	@Override
	public void dropFeatureType(String typeName) throws IOException, TypeNotFoundException 
	{
		String uuid = getLock();
		try
		{
			super.dropFeatureType(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected TerralibAttributeDescriptor buildTerralibAttributeDescriptor(
			String name, boolean isNullable, boolean isPrimaryKey, int length,
			TeAttrDataType type) {
		String uuid = getLock();
		try
		{
			return super.buildTerralibAttributeDescriptor(name, isNullable, isPrimaryKey,
					length, type);
		}
		finally
		{
			releaseLock(uuid);
		}
		
	}

	@Override
	protected void createAttributeTable(String tableName,
			TerralibAttributeDescriptorVector attributes)
			throws IllegalStateException, IOException {
		String uuid = getLock();
		try
		{
			super.createAttributeTable(tableName, attributes);
		}
		finally
		{
			releaseLock(uuid);
		}
		
	}

	@Override
	protected void createType(String typeName, TeGeomRep geomRep, String crsWkt)
			throws IOException, InvalidCrsWktException {
		String uuid = getLock();
		try
		{
			super.createType(typeName, geomRep, crsWkt);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	public void dispose() {
		String uuid = getLock();
		try
		{
			super.dispose();
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	public void dropFeatureType(String typeName, boolean ignoreAttributes) throws TypeNotFoundException, IOException {
		String uuid = getLock();
		try
		{
			super.dropFeatureType(typeName, ignoreAttributes);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected Envelope getBoundingBox(String typeName)
			throws IllegalStateException, TypeNotFoundException {
		String uuid = getLock();
		try
		{
			return super.getBoundingBox(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	public int getGeometryCount(String typeName) throws TypeNotFoundException,
			IOException {
		String uuid = getLock();
		try
		{
			return super.getGeometryCount(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
		
	}

	@Override
	protected int getGeomRep(String typeName) throws IllegalStateException,
			TypeNotFoundException {
		String uuid = getLock();
		try
		{
			return super.getGeomRep(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected TerralibAttributeDescriptorVector getTypeAttributes(
			String typeName) throws IllegalStateException,
			TypeNotFoundException, IOException {
		String uuid = getLock();
		try
		{
			return super.getTypeAttributes(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected String getTypeProjectionWKT(String typeName)
			throws IllegalStateException, TypeNotFoundException {
		String uuid = getLock();
		try
		{
			return super.getTypeProjectionWKT(typeName);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	public void getTypesNames(StringVector out) throws IllegalStateException {
		String uuid = getLock();
		try
		{
			super.getTypesNames(out);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected void getViewIDs(StringVector out) {
		String uuid = getLock();
		try
		{
			super.getViewIDs(out);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected String getViewProjectionWKT(String viewID)
			throws TypeNotFoundException {
		String uuid = getLock();
		try
		{
			return super.getViewProjectionWKT(viewID);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	protected void getViewTypesNames(String viewID, StringVector out) {
		String uuid = getLock();
		try
		{
			super.getViewTypesNames(viewID, out);
		}
		finally
		{
			releaseLock(uuid);
		}
		
	}

	@Override
	public String insert(String typeName,
			PersistenceTransferenceObject transferObject,
			boolean ignoreAttributes) throws IllegalStateException,
			TypeNotFoundException, IOException {
		//getLock();
		try
		{
			return super.insert(typeName, transferObject, ignoreAttributes);
		}
		finally
		{
			//releaseLock();
		}
			
	}

	@Override
	public String insert(String typeName,
			PersistenceTransferenceObject transferObject)
			throws IllegalStateException, TypeNotFoundException, IOException {
		//getLock();
		try
		{
			return super.insert(typeName, transferObject);
		}
		finally
		{
			//releaseLock();
		}
	}

	@Override
	protected void linkAttributeTable(String typeName, String tableName,
			String linkAttributeColumn) throws TypeNotFoundException,
			IllegalStateException, IOException {
		String uuid = getLock();
		try
		{
			super.linkAttributeTable(typeName, tableName, linkAttributeColumn);
		}
		finally
		{
			releaseLock(uuid);
		}
	}

	@Override
	public void remove(String typeName, String featureId,
			boolean ignoreAttributes) throws IllegalStateException,
			TypeNotFoundException, IOException {
		//getLock();
		try
		{
			super.remove(typeName, featureId, ignoreAttributes);
		}
		finally
		{
			//releaseLock();
		}
	}

	@Override
	public void remove(String typeName, String featureId)
			throws IllegalStateException, TypeNotFoundException, IOException {
		//getLock();
		try
		{
			super.remove(typeName, featureId);
		}
		finally
		{
			//releaseLock();
		}
		
		
	}
	
	@Override
	public void removeAllFeatures(String typeName, boolean ignoreAttributes) throws IllegalStateException, TypeNotFoundException, IOException {
		try
		{
			super.removeAllFeatures(typeName, ignoreAttributes);
		}
		finally
		{
			
		}
	}

	@Override
	public void update(String typeName,
			PersistenceTransferenceObject transferObject,
			boolean ignoreAttributes) throws IllegalStateException,
			TypeNotFoundException, IOException {
		//getLock();
		try
		{
			super.update(typeName, transferObject, ignoreAttributes);
		}
		finally
		{
			//releaseLock();
		}
	}

	@Override
	public void update(String typeName,
			PersistenceTransferenceObject transferObject)
			throws IllegalStateException, TypeNotFoundException, IOException {
		//getLock();
		try
		{
			super.update(typeName, transferObject);
		}
		finally
		{
			//releaseLock();
		}
		
	}
	
	
	public String getLock()
	{
		StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
		String uuid = null;
		long threadID = Thread.currentThread().getId();
		if(_enableLock)
		{
			synchronized (this) {
				if(_semaphore.availablePermits() == 0)
				{
					uuid = UUID.randomUUID().toString();
					_watchdog.add(uuid,threadID,stackTrace,_lastSuccessfullThreadID,_lastSuccessfullCallerStack, WATCHDOG_TIMEOUT_MILLISECONDS);
				}
			}
			_semaphore.acquireUninterruptibly();
			synchronized(this)
			{
				_lastSuccessfullCallerStack = stackTrace;
				_lastSuccessfullThreadID = threadID;  
			}
			
		}
		return uuid;
		
	}

	public void releaseLock(String uuid)
	{
		if(_enableLock)
		{
			_semaphore.release();
			synchronized(this)
			{
				_lastSuccessfullCallerStack = EMPTY_STACK;
				_lastSuccessfullThreadID = NO_THREAD;
			}
			_watchdog.remove(uuid);
		}
	}
	
	

}
