/**
 * Tecgraf - GIS development team
 *
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 *
 * file: DBConnectionParamUtils.java
 * created: 22/12/2008
 */
package org.geotools.data.terralib.connection;

import java.io.File;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.geotools.data.Parameter;
import org.geotools.data.DataAccessFactory.Param;
import org.geotools.util.SimpleInternationalString;

/**
 * This is an utility class used to support the usage of parameter maps with the TerralibDataStore.
 * @author alexmc
 * @since TDK 3.0.0
 */
public class DBConnectionParamUtils
{
    public static final String DATABASE_DRIVER_ACCESS = "Access";
    public static final String DATABASE_DRIVER_SQL_SERVER = "SQLServer";
    public static final String TERRALIB_DATABASE_TYPE = "terralib";


    public static final Param DBTYPE = new Param("dbtype", String.class,"must be 'terralib'", true, TERRALIB_DATABASE_TYPE);

    public static final Param DBDRIVER = new Param("dbdriver", String.class,"database driver to use (Access or SQLServer)", true, DATABASE_DRIVER_ACCESS);

    public static final Param DBPATH = new Param("dbpath", String.class,"terralib access database path", false);

    public static final Param HOST = new Param("host", String.class,"terralib host machine", false, "localhost");

    public static final Param USER = new Param("user", String.class, "user name to login as",false,"admin");

    public static final Param PASSWD = new Param("passwd", String.class,new SimpleInternationalString("password used to login"),
            false, null, Collections.singletonMap(Parameter.IS_PASSWORD, Boolean.TRUE));

    public static final Param DBNAME = new Param("dbname", String.class,"database name",false);

    public static final Param PORT = new Param("port", Integer.class, "terralib connection port (default is 5432)", false
            , Integer.valueOf(5432));  //new Integer(5432)


    public static final Integer DEFAULT_PORT_NUMBER = -1;

    private static final List<String> _validDatabaseTypes;
    static
    {
        _validDatabaseTypes = Collections.unmodifiableList( Arrays.asList(DATABASE_DRIVER_ACCESS, DATABASE_DRIVER_SQL_SERVER) );
    }

    public static Map<String, ? extends Serializable> createAccessParametersMap(String path)
    {
        Map<String, Serializable> parameterMap = new HashMap<String, Serializable>();

        parameterMap.put(DBDRIVER.key, DATABASE_DRIVER_ACCESS);
        parameterMap.put(DBPATH.key, path);
        parameterMap.put(DBTYPE.key, TERRALIB_DATABASE_TYPE);

        checkParams(parameterMap);

        return parameterMap;
    }

    public static Map<String, ? extends Serializable> createParametersMap(String dataBaseType, String host, String dataBaseName, String userName, String password)
    {
        return createParametersMap(dataBaseType, host, DEFAULT_PORT_NUMBER, dataBaseName, userName, password);
    }

    /*
     * TODO:
     */
    public static Map<String, ? extends Serializable> createParametersMap(String dataBaseDriver, String host, Integer portNumber, String dataBaseName, String userName, String password)
    {
        Map<String, Serializable> parameterMap = new HashMap<String, Serializable>();

        parameterMap.put(DBDRIVER.key, dataBaseDriver);
        parameterMap.put(HOST.key, host);
        parameterMap.put(DBNAME.key, dataBaseName);
        parameterMap.put(PORT.key, portNumber);
        parameterMap.put(USER.key, userName);
        parameterMap.put(PASSWD.key, password);
        parameterMap.put(DBTYPE.key, TERRALIB_DATABASE_TYPE);

        checkParams(parameterMap);

        return parameterMap;
    }

    /**
     * Checks if the parameter map is valid TerralibDataStore parameter map. If the map is invalid,
     * an exception is thrown.
     * @param paramMap The parameter map to be checked.
     * @throws IllegalArgumentException If the parameter map is invalid.
     */
    public static void checkParams(Map<String, ? extends Serializable> paramMap)
    {
        String databaseTypeName = (String) paramMap.get(DBDRIVER.key);

        checkParamIsNotNull(DBTYPE.key, (String) paramMap.get(DBTYPE.key));

        if (!TERRALIB_DATABASE_TYPE.equalsIgnoreCase((String)paramMap.get(DBTYPE.key)))
            throw new IllegalArgumentException("The database type must be " + TERRALIB_DATABASE_TYPE + ".");

        if (null == databaseTypeName)
        {
            throw new IllegalArgumentException("The database driver can't be null.");
        }
        else if (!_validDatabaseTypes.contains(databaseTypeName))
        {
            throw new IllegalArgumentException("Unknown database driver: " + databaseTypeName);
        }

        if ( DATABASE_DRIVER_ACCESS.equalsIgnoreCase(databaseTypeName) )
        {
            checkAccessParams(paramMap);
        }
        else
        {
            checkRegularParams(paramMap);
        }
    }

    /**
     * Checks the needed parameters for a non-access (regular) terralib database. It's assumed that
     * the database type name has been already checked and is correct.
     * @param paramMap Parameter map we will check.
     */
    private static void checkRegularParams(Map<String, ? extends Serializable> paramMap)
    {
        String host = (String) paramMap.get(HOST.key);
        checkParamIsNotNull(HOST.key, host);
        checkParamIsNotEmpty(HOST.key, host);

        String user = (String) paramMap.get(USER.key);
        checkParamIsNotNull(USER.key, user);
        checkParamIsNotEmpty(USER.key, user);

        String password = (String) paramMap.get(PASSWD.key);
        checkParamIsNotNull(PASSWD.key, password);
        checkParamIsNotEmpty(PASSWD.key, password);

        String databaseName = (String) paramMap.get(DBNAME.key);
        checkParamIsNotNull(DBNAME.key, databaseName);
        checkParamIsNotEmpty(DBNAME.key, databaseName);

        Serializable port = paramMap.get(PORT.key);
        checkParamIsNotNull(PORT.key, port);
        Integer portNumber = Integer.parseInt(port.toString());
        if (portNumber.compareTo(DEFAULT_PORT_NUMBER) != 0)
        {
            checkParamIsNotNegative(PORT.key, portNumber);
        }
    }

    /**
     * Checks the needed parameters for an access database. It's assumed that
     * the database type name has been already checked and is correct.
     * @param paramMap Parameter map we will check.
     */
    private static void checkAccessParams(Map<String, ? extends Serializable> paramMap)
    {
        String path = (String) paramMap.get(DBPATH.key);

        checkParamIsNotNull(DBPATH.key, path);
        checkParamIsNotEmpty(DBPATH.key, path);

        File databaseFile = new File(path);

        if ( !databaseFile.exists() || !databaseFile.isFile() )
        {
            throw new IllegalArgumentException("Path doesn't point to a valid file.");
        }

        String fileName = databaseFile.getName();

        if ( !fileName.toLowerCase().endsWith(".mdb") )
        {
            throw new IllegalArgumentException("The referenced file hasn't an access file extension.");
        }
    }

    /**
     * Checks if the given object parameter value is not null or throws otherwise.
     * @param parameterName Name of the parameter, for the descriptive message.
     * @param parameterValue Value of the parameter (what we'll check).
     */
    private static void checkParamIsNotNull(String parameterName, Object parameterValue)
    {
        assert(parameterName != null);

        if (null == parameterValue)
        {
            throw new IllegalArgumentException("Parameter \"" + parameterName + "\" cannot be empty.");
        }
    }

    /**
     * Checks if the given string parameter value is not null or throws otherwise.
     * @param parameterName Name of the parameter, for the descriptive message.
     * @param parameterValue Value of the parameter (what we'll check).
     */
    private static void checkParamIsNotEmpty(String parameterName, String parameterValue)
    {
        assert(parameterName != null);

        if (parameterValue.isEmpty())
        {
            throw new IllegalArgumentException("Parameter \"" + parameterName + "\" cannot be empty.");
        }
    }

    /**
     * Checks if the given number parameter value non negative or throws otherwise.
     * @param parameterName Name of the parameter, for the descriptive message.
     * @param parameterValue Value of the parameter (what we'll check).
     */
    private static void checkParamIsNotNegative(String parameterName, Number parameterValue)
    {
        assert(parameterName != null);

        if (parameterValue.doubleValue() < 0.0)
        {
            throw new IllegalArgumentException("Parameter \"" + parameterName + "\" cannot be negative.");
        }
    }
}
