package org.geotools.data.terralib;

import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Set;


import org.geotools.data.terralib.exception.FeatureNotFoundException;
import org.geotools.data.terralib.exception.IllegalStateException;
import org.geotools.data.terralib.exception.InvalidCrsWktException;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.persistence.exception.InvalidAttributeException;
import org.geotools.data.terralib.persistence.exception.InvalidAttributeValueException;
import org.geotools.data.terralib.persistence.exception.MissingRequiredAttributeException;
import org.geotools.data.terralib.swig.PersistenceTransferenceObject;
import org.geotools.data.terralib.swig.TeAttrDataType;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptor;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

/**
 * This interface represents the classes that intend to be the access point to Terralib native services.
 * @author fmoura, fabiomano
 * @since TDK 3.0.0
 */
public interface TerralibService
{
    /**
     * Gets a GeoAPI {@link CoordinateReferenceSystem} that represents the coordinate reference system 
     * for a given typeName.
     * @param typeName A valid {@link TerralibDataStore} type name that identifies the type witch the CRS will be retrieved. 
     * @return The given type CRS.
     * @since TDK 3.0.0
     * @throws IllegalStateException if the database is in a invalid state ( commonly not connected ).
     * @throws IllegalArgumentException if the type name does not conform to {@link TerralibDataStore} type name convention.
     * @throws TypeNotFoundException if the given type is not found in the working database.
     * @throws IOException if data connot be retrieved from the database.
     */
    public CoordinateReferenceSystem getTypeCRS(String typeName) throws IllegalStateException,TypeNotFoundException, IOException;

    /**
     * Retrieves the set of {@link GeometryDescriptor} present in the given feature type
     * @param typeName A valid {@link TerralibDataStore} type name that identifies the type for witch the geometry descriptors will be retrieved
     * @return a set of {@link GeometryDescriptor} that describes the geometry attributes retrieved for this feature type.
     * @throws IllegalStateException if the database is in a invalid state ( commonly not connected ).
     * @throws IllegalArgumentException if the type name does not conform to {@link TerralibDataStore} type name convention.
     * @throws TypeNotFoundException if the given type is not found in the working database.
     * @throws IOException if data connot be retrieved from the database.
     * @since TDK 3.0.0
     */
    public Set<GeometryDescriptor> getTypeGeometryDescriptors(String typeName) throws IllegalStateException, TypeNotFoundException , IOException;
    
    /**
     * Gets the {@link AttributeDescriptor} lists, indexed by the Terralib attribute table names, associated with the
     * given feature type.
     * The Terralib feature types ( layers and themes ) can be joint to one or more attribute tables. 
     * @param typeName A valid {@link TerralibDataStore} type name that identifies the type for witch the geometry descriptors will be retrieved 
     * @return A Map<String,List<AttributeDescriptor>> that contains all non geometry type attributes for the given feature type, grouped by attribute tables
     * @throws IllegalStateException if the database is in a invalid state ( commonly not connected ).
     * @throws IllegalArgumentException if the type name does not conform to {@link TerralibDataStore} type name convention.
     * @throws TypeNotFoundException if the given type is not found in the working database.
     * @throws IOException if data cannot be retrieved from the database.
     * @since TDK 3.0.0
     */
    public Map<String,List<AttributeDescriptor>> getTypeAttributeDescriptors(String typeName) throws IllegalStateException, TypeNotFoundException, IOException;
    
    
    /**
     * Gets a {@link RefencedEnvelope} that represents the entire type bounding box
     * @param typeName The type name.
     * @return The Type complete bounding box.
     * @throws MismatchedDimensionException if the retrieved bounding box does not match the retrieved reference system. 
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws TypeNotFoundException if the given type name does not match any type in the terralib datastore.
     * @throws IOException if data cannot be retrieved from the database. 
     * @since TDK 3.0.0
     */
    public ReferencedEnvelope getTypeBoundingBox(String typeName) throws MismatchedDimensionException, IllegalStateException, TypeNotFoundException, IOException;

    
    /**
     * Updates the given tuple at the database.
     * @param typeName the type identifier where to update the given tuple.
     * @param transferObject the object that represents the tuple to be updated.
     * @param ignoreAttributes indicates that only the geometries in the tranferObject should be persisted.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws MissingRequiredAttributeException if a required attribute is missing at the transfer object.
     * @throws InvalidAttributeException if a given attribute does not exists in the type schema.
     * @throws InvalidAttributeValueException if a given attribute value is incompatible with it's type.
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws IOException if an error occurred while trying to persist the data.
     * @throws FeatureNotFoundException if the given tuple identifier is not found.
     * @throws NullArgumentException if the given {@link PersistenceTransferenceObject} is null.
     * @since TDK 3.0.0 
     */
    public void update(String typeName,
            PersistenceTransferenceObject transferObject, boolean ignoreAttributes) throws IllegalStateException, TypeNotFoundException, IOException;  
    
    /**
     * Updates the given tuple at the database.
     * @param typeName the type identifier where to update the given tuple.
     * @param transferObject the object that represents the tuple to be updated.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws MissingRequiredAttributeException if a required attribute is missing at the transfer object.
     * @throws InvalidAttributeException if a given attribute does not exists in the type schema.
     * @throws InvalidAttributeValueException if a given attribute value is incompatible with it's type.
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws IOException if an error occurred while trying to persist the data.
     * @throws FeatureNotFoundException if the given tuple identifier is not found.
     * @throws NullArgumentException if the given {@link PersistenceTransferenceObject} is null.
     * @since TDK 3.0.0 
     */
    public void update(String typeName,
            PersistenceTransferenceObject transferObject) throws IllegalStateException, TypeNotFoundException, IOException;  
    
    /**
     * Inserts the given tuple at the database.
     * @param typeName the type identifier where to insert the given tuple.
     * @param transferObject the object that represents the tuple to be inserted.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws MissingRequiredAttributeException if a required attribute is missing at the transfer object.
     * @throws InvalidAttributeException if a given attribute does not exists in the type schema.
     * @throws InvalidAttributeValueException if a given attribute value is incompatible with it's type.
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws IOException if an error occurred while trying to persist the data.
     * @throws NullArgumentException if the given {@link PersistenceTransferenceObject} is null.
     * @since TDK 3.0.0
     */
    public String insert(String typeName,
            PersistenceTransferenceObject transferObject) throws IllegalStateException, TypeNotFoundException, IOException;
    
    /**
     * Inserts the given tuple at the database.
     * @param typeName the type identifier where to insert the given tuple.
     * @param transferObject the object that represents the tuple to be inserted.
     * @param ignoreAttributes indicates that only the geometries in the tranferObject should be persisted. No attribute check is made.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws MissingRequiredAttributeException if a required attribute is missing at the transfer object.
     * @throws InvalidAttributeException if a given attribute does not exists in the type schema.
     * @throws InvalidAttributeValueException if a given attribute value is incompatible with it's type.
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws IOException if an error occurred while trying to persist the data.
     * @throws NullArgumentException if the given {@link PersistenceTransferenceObject} is null.
     * @since TDK 3.0.0
     */
    public String insert(String typeName,
            PersistenceTransferenceObject transferObject, boolean ignoreAttributes) throws IllegalStateException, TypeNotFoundException, IOException;
    
    
    /**
     * Removes a feature from the database. 
     * @param typeName The type identifier from where to remove the feature.
     * @param featureId The featureId that identifies the feature to be removed.
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws IOException if an error occurred while trying to remove the data.
     * @since TDK 3.0.0
     */
    public void remove(String typeName, String featureId)  throws IllegalStateException, TypeNotFoundException, IOException; 
    
    /**
     * Removes a feature from the database.
     * @param typeName The type identifier from where to remove the feature.
     * @param featureId The featureId that identifies the feature to be removed.
     * @param ignoreAttributes indicates that only the geometries should be removed from the database. 
     * @throws IllegalStateException if the terralib database connection is in a invalid state.
     * @throws TypeNotFoundException if no type is found for the given type name.
     * @throws IOException if an error occurred while trying to remove the data.
     * @since TDK 3.0.0
     */
    public void remove(String typeName, String featureId, boolean ignoreAttributes)  throws IllegalStateException, TypeNotFoundException, IOException;
    
    /**
     * Retrieves a list of of the available FeatureTypes.
     * <p>
     * This is simply a list of the FeatureType names as aquiring the actual
     * FeatureType schemas may be expensive.
     * </p>
     * @return typeNames for available FeatureTypes.
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws IOException if the data cannot be retrieved.
     */    
    public String[] getTypeNames() throws IllegalStateException, IOException;
    
    /**
     * Gets the list view ids at the terralib database. Only gets the views that
     * are used in a theme.
     * @return the list view ids at the terralib database.
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws IOException if the data cannot be retrieved.
     */
    public String[] getViewIDs() throws IllegalStateException, IOException;
    
    /**
     * Gets the view name for a given id
     * @param viewID The view ID to retrieve the name
     * @return The view name for a given id
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws TypeNotFoundException if the view does not exist.
     */
    public String getViewName(String viewID) throws IllegalStateException, TypeNotFoundException;
    
    /**
     * Gets the list type names at a given view.
     * @return the list type names at a given view.
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws IOException if the data cannot be retrieved.
     */
    public String[] getViewTypesNames(String viewID) throws IllegalStateException , IOException;
    
    /**
     * Gets the view Coordinate Reference System.
     * @param viewID The ID of the view to get the CRS.
     * @throws TypeNotFoundException When the view id is not found.
     * @return the view CRS.
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws IOException if the data cannot be retrieved.
     */
    public CoordinateReferenceSystem getViewCRS(String viewID) throws TypeNotFoundException, IllegalStateException, IOException;    


    /**
     * Creates a new layer at the terraLib database.
     * @param typeName The layer name
     * @param desc Geometry type for this layer
     * @param projection The layer's Coordinate Reference System
     * @throws IllegalStateException if the database connection is in a invalid state.
     * @throws InvalidCrsWktException
     */
    public void createType(String typeName, GeometryDescriptor geometryDesc, CoordinateReferenceSystem projection) throws IllegalStateException,InvalidCrsWktException, IOException;


    /**
     * Creates a new Attribute table at the terralib database.
     * This method will create the table at the database, with the fields described
     * at the attributs list. 
     * @param tableName The table name
     * @param attributes List of attributes that this table will hold.
     * @throws IOException if an error occurs during table creation on the database
     * @throws IllegalStateException if the database connection is in a invalid state
     */
    public void createAttributeTable(String tableName, List<AttributeDescriptor> attributes) throws IOException, IllegalStateException;
    
    /**
     * Links an attribute table to a terralib layer (type).
     * @param typeName The type name to associate the attribute table.
     * @param tableName The attribute table name 
     * @param linkAttributeColumn The attribute table's column name that will be linked to the type's object_id
     * @throws TypeNotFoundException if the given typeName is not found at the database.
     * @throws IOException if an error occurs during table creation on the database
     * @throws IllegalStateException if the database connection is in a invalid state
     */
    public void linkAttributeTable(String typeName, String tableName, String linkAttributeColumn) throws IOException, TypeNotFoundException, IllegalStateException;
    
    /**
     * Creates a new TerralibAttributeDescriptor
     * @param name Name of the attribute
     * @param isNullable If the attribute can be null
     * @param isPrimaryKey If the attribute is a primary key
     * @param length Length of the attribute. Use static method getANY_LENGHT() to accept any length for the attribute.
     * @param type Data type of the attribute 
     * @return The TerralibAttributeDescriptor created.
     */
    public TerralibAttributeDescriptor buildTerralibAttributeDescriptor(String name, boolean isNullable, boolean isPrimaryKey, int length, TeAttrDataType type);
    
    /**
     * Locks any other subsequent terralib service calls until releaseLock is called. 
     * It internally uses a binary semaphore.
     * @return The unique identifier for this lock  
     */
    public String getLock();
    
    /**
     * Releases a previous acquired lock for this terralib service.
     * @param The unique identifier of the lock to release.
     */
    public void releaseLock(String uuid);
    
    /**
     * Remove all features from a given type.
     * @param typeName The feature type name.
     * @param ignoreAttributes indicates that only the geometries should be removed from the database.
     * @throws IllegalStateException if the database connection is in a invalid state
     * @throws TypeNotFoundException if the given typeName is not found at the database.
     * @throws IOException if an error occurs during feature removal.
     */
    public void removeAllFeatures(String typeName, boolean ignoreAttributes) throws IllegalStateException , TypeNotFoundException, IOException;
    
    /**
	 * Gets the data type for a given layer. Terralib accepts a type with both Coverage and Vectorial datas,
	 * but we haven't got any real example with this.
	 * @param typeName The type name to check
	 * @return the {@link TerralibLayerDataType}
	 * @throws IOException if any error occurs while retrieving this info.
     * @throws TypeNotFoundException if the given type was not found.
     * @throws IllegalStateException if the database connection is at an illegal state.
     */
    public TerralibLayerDataType getLayerType(String typeName) throws IllegalStateException, TypeNotFoundException, IOException;
    
    /**
     * Retrieves the list of coverage files associated with the given type
	 * @param typeName
	 * @return A {@link List} of {@link URL} which locates the coverage files associated with the given type.
	 * @throws IOException if any error occurs while retrieving this info.
     * @throws TypeNotFoundException if the given type was not found.
     * @throws IllegalStateException if the database connection is at an illegal state.
     */
    public List<URL> getRasterFileURL(String typeName) throws TypeNotFoundException, IllegalStateException, IOException;
    
}
