package tecgraf.openbus.assistant;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.omg.CORBA.COMM_FAILURE;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;
import org.omg.CORBA.TRANSIENT;
import org.omg.CORBA.ORBPackage.InvalidName;

import scs.core.IComponent;
import tecgraf.openbus.Connection;
import tecgraf.openbus.InvalidLoginCallback;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.PrivateKey;
import tecgraf.openbus.core.ORBInitializer;
import tecgraf.openbus.core.v2_0.OctetSeqHolder;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidLoginCode;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.access_control.LoginProcess;
import tecgraf.openbus.core.v2_0.services.access_control.MissingCertificate;
import tecgraf.openbus.core.v2_0.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidProperties;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidService;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistry;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOffer;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;
import tecgraf.openbus.core.v2_0.services.offer_registry.UnauthorizedFacets;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.exception.InvalidLoginProcess;
import tecgraf.openbus.exception.InvalidPropertyValue;

/**
 * Assistente que auxilia a integrao de uma aplicao a um barramento.
 * <p>
 * O assistente realiza tarefas de manuteno da integrao da aplicao com o
 * barramento. Tais tarefas incluem:
 * <ul>
 * <li>Restabelecimento de login, mesmo em virtude de falhas temporrias.
 * <li>Registro de ofertas e observadores de servio, mesmo aps
 * restabelecimento de login.
 * <li>Busca de ofetas de servio disponveis no barramento.
 * </ul>
 * <p>
 * O assistente implementa a mesma interface do registro de ofertas, porm nunca
 * lana excees, pois todas as operaes so realizadas de forma assncrona.
 * Eventuais falhas nessas operaes assncronas so notificadas atravs de
 * callbacks.
 * 
 * @author Tecgraf
 */
public abstract class Assistant {

  /** Instncia do logger */
  private static final Logger logger = Logger.getLogger(Assistant.class
    .getName());

  /** Intervalo de espera entre tentativas */
  private int interval = 5;
  /** Host com o qual o assistente quer se conectar */
  private String host;
  /** Porta com a qual o assistente quer se conectar */
  private int port;
  /** ORB utilizado pelo assistente */
  private ORB orb;
  /** Contexto do ORB utilizado */
  private OpenBusContext context;
  /** Conexo criada e utilizada pelo assistente */
  private Connection conn;
  /** Callback para informar os erros ocorridos no uso do assistente */
  private OnFailureCallback callback;
  /** Lista de ofertas a serem mantidas pelo assistente */
  private List<Offer> offers = Collections
    .synchronizedList(new ArrayList<Offer>());
  /** Identifica se o assistente deve finalizar */
  private volatile boolean shutdown = false;

  /** Controlador do pool de threads utilizadas pelo assistente */
  private ExecutorService threadPool = Executors
    .newCachedThreadPool(new ThreadFactory() {
      /**
       * Cria threads Daemon para serem utilizadas pelo {@link ExecutorService}.
       */
      @Override
      public Thread newThread(Runnable task) {
        Thread thread = new Thread(task);
        thread.setDaemon(true);
        return thread;
      }
    });

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * definida pelo mtodo {@link Assistant#onLoginAuthentication()}.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao definida pelo mtodo
   * {@link Assistant#onLoginAuthentication()} que informa a forma de
   * autenticao, assim como os dados para realizar essa autenticao.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   */
  public Assistant(String host, int port) {
    this(host, port, null);
  }

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * definida pelo mtodo {@link Assistant#onLoginAuthentication()}.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao definida pelo mtodo
   * {@link Assistant#onLoginAuthentication()} que informa a forma de
   * autenticao, assim como os dados para realizar essa autenticao.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param params Parmetros opicionais de configurao do assistente
   */
  public Assistant(String host, int port, AssistantParams params) {
    this.host = host;
    this.port = port;
    if (params == null) {
      params = new AssistantParams();
    }
    orb = params.orb;
    if (orb == null) {
      orb = ORBInitializer.initORB();
    }
    try {
      this.context =
        (OpenBusContext) orb.resolve_initial_references("OpenBusContext");
    }
    catch (InvalidName e) {
      throw new IllegalArgumentException(
        "ORB utilizado no foi inicializado corretamente.", e);
    }
    try {
      this.conn = this.context.createConnection(host, port, params.connprops);
    }
    catch (InvalidPropertyValue e) {
      throw new IllegalArgumentException(
        "Propriedades definidas para a conexo so invlidas", e);
    }
    if (context.getDefaultConnection() != null) {
      throw new IllegalArgumentException("ORB j est em uso.");
    }
    context.setDefaultConnection(conn);
    conn.onInvalidLoginCallback(new OnInvalidLogin());
    if (params.interval != null) {
      interval = params.interval;
    }
    if (params.callback != null) {
      this.callback = params.callback;
    }
    else {
      this.callback = new DefaultFailureCallback();
    }
    // realiza o login
    threadPool.execute(new DoLogin(this));
  }

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * por senha.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao da entidade indicada pelo parmetro 'entity' e a
   * senha fornecida pelo parmetro 'password'.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param entity Identificador da entidade a ser autenticada.
   * @param password Senha de autenticao no barramento da entidade.
   * 
   * @return um novo assistente.
   */
  public static Assistant createWithPassword(String host, int port,
    final String entity, final byte[] password) {
    return createWithPassword(host, port, entity, password, null);
  }

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * por senha.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao da entidade indicada pelo parmetro 'entity' e a
   * senha fornecida pelo parmetro 'password'.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param entity Identificador da entidade a ser autenticada.
   * @param password Senha de autenticao no barramento da entidade.
   * @param params Parmetros opicionais de configurao do assistente
   * 
   * @return um novo assistente.
   */
  public static Assistant createWithPassword(String host, int port,
    final String entity, final byte[] password, AssistantParams params) {
    return new Assistant(host, port, params) {

      @Override
      public AuthArgs onLoginAuthentication() {
        return new AuthArgs(entity, password);
      }
    };
  }

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * por certificado.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao da entidade indicada pelo parmetro 'entity' e a
   * chave privada fornecida pelo parmetro 'key'.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param entity Identificador da entidade a ser autenticada.
   * @param key Chave privada correspondente ao certificado registrado a ser
   *        utilizada na autenticao.
   * 
   * @return um novo assistente.
   */
  public static Assistant createWithPrivateKey(String host, int port,
    final String entity, final PrivateKey key) {
    return createWithPrivateKey(host, port, entity, key, null);
  }

  /**
   * Cria um assistente que efetua login no barramento utilizando autenticao
   * por certificado.
   * <p>
   * Assistentes criados com essa operao realizam o login no barramento sempre
   * utilizando autenticao da entidade indicada pelo parmetro 'entity' e a
   * chave privada fornecida pelo parmetro 'key'.
   * 
   * @param host Endereo ou nome de rede onde os servios ncleo do barramento
   *        esto executando.
   * @param port Porta onde os servios ncleo do barramento esto executando.
   * @param entity Identificador da entidade a ser autenticada.
   * @param key Chave privada correspondente ao certificado registrado a ser
   *        utilizada na autenticao.
   * @param params Parmetros opicionais de configurao do assistente
   * 
   * @return um novo assistente.
   */
  public static Assistant createWithPrivateKey(String host, int port,
    final String entity, final PrivateKey key, AssistantParams params) {
    return new Assistant(host, port, params) {

      @Override
      public AuthArgs onLoginAuthentication() {
        return new AuthArgs(entity, key);
      }
    };
  }

  /**
   * Consulta o ORB em utilizado pelo assistente.
   * 
   * @return ORB utilizado pelo assistante.
   */
  public ORB orb() {
    return this.orb;
  }

  /**
   * Solicita que o assitente registre um servio no barramento.
   * <p>
   * Esse mtodo notifica o assistente de que o servio fornecido deve se
   * mantido como uma oferta de servio vlida no barramento. Para tanto, sempre
   * que o assistente restabelecer o login esse servio ser novamente
   * registrado no barramento.
   * <p>
   * Para que o registro de servios seja bem sucedido  necessrio que o ORB
   * utilizado pelo assistente esteja processando chamadas, por exemplo, fazendo
   * com que a aplicao chame o mtodo 'ORB::run()'.
   * <p>
   * Caso ocorram erros, a callback de tratamento de erro apropriada ser
   * chamada.
   * 
   * @param component Referncia do servio sendo ofertado.
   * @param properties Propriedades do servio sendo ofertado.
   */
  public void registerService(IComponent component, ServiceProperty[] properties) {
    Offer offer = new Offer(this, component, properties);
    this.offers.add(offer);
    // dispara o registro da oferta de servio
    threadPool.execute(new DoRegister(this, offer));
  }

  /**
   * Busca por ofertas que apresentem um conjunto de propriedades definido.
   * <p>
   * Sero selecionadas apenas as ofertas de servio que apresentem todas as
   * propriedades especificadas. As propriedades utilizadas nas buscas podem ser
   * aquelas fornecidas no momento do registro da oferta de servio, assim como
   * as propriedades automaticamente geradas pelo barramento.
   * <p>
   * Caso ocorram erros, a callback de tratamento de erro apropriada ser
   * chamada. Se o nmero de tentativas se esgotar e no houver sucesso, uma
   * sequncia vazia ser retornada.
   * 
   * @param properties Propriedades que as ofertas de servios encontradas devem
   *        apresentar.
   * @param retries Parmetro opcional indicando o nmero de novas tentativas de
   *        busca de ofertas em caso de falhas, como o barramento estar
   *        indisponvel ou no for possvel estabelecer um login at o momento.
   *        'retries' com o valor 0 implica que a operao retorna imediatamente
   *        aps uma nica tentativa. Para tentar indefinidamente o valor de
   *        'retries' deve ser -1. Entre cada tentativa  feita uma pausa dada
   *        pelo parmetro 'interval' fornecido na criao do assistente (veja a
   *        interface 'AssistantFactory').
   * 
   * @return Sequncia de descries de ofertas de servio encontradas.
   * @throws Throwable
   */
  public ServiceOfferDesc[] findServices(ServiceProperty[] properties,
    int retries) throws Throwable {
    int attempt = retries;
    Throwable last;
    do {
      last = null;
      if (conn.login() != null) {
        try {
          ServiceOfferDesc[] offerDescs = find(properties);
          if (offerDescs != null) {
            return offerDescs;
          }
        }
        catch (Throwable e) {
          last = e;
        }
      }
    } while (shouldRetry(retries, --attempt));
    if (last != null) {
      throw last;
    }
    return new ServiceOfferDesc[0];
  }

  /**
   * Devolve uma lista de todas as ofertas de servio registradas.
   * <p>
   * Caso ocorram erros, a callback de tratamento de erro apropriada ser
   * chamada. Se o nmero de tentativas se esgotar e no houver sucesso, uma
   * sequncia vazia ser retornada.
   * 
   * @param retries Parmetro opcional indicando o nmero de novas tentativas de
   *        busca de ofertas em caso de falhas, como o barramento estar
   *        indisponvel ou no for possvel estabelecer um login at o momento.
   *        'retries' com o valor 0 implica que a operao retorna imediatamente
   *        aps uma nica tentativa. Para tentar indefinidamente o valor de
   *        'retries' deve ser -1. Entre cada tentativa  feita uma pausa dada
   *        pelo parmetro 'interval' fornecido na criao do assistente (veja a
   *        interface 'AssistantFactory').
   * 
   * @return Sequncia de descries de ofertas de servio registradas.
   * @throws Throwable
   */
  public ServiceOfferDesc[] getAllServices(int retries) throws Throwable {
    int attempt = retries;
    Throwable last;
    do {
      last = null;
      if (conn.login() != null) {
        try {
          ServiceOfferDesc[] offerDescs = getAll();
          if (offerDescs != null) {
            return offerDescs;
          }
        }
        catch (Throwable e) {
          last = e;
        }
      }
    } while (shouldRetry(retries, --attempt));
    if (last != null) {
      throw last;
    }
    return new ServiceOfferDesc[0];
  }

  /**
   * Inicia o processo de login por autenticao compartilhada.
   * <p>
   * A autenticao compartilhada permite criar um novo login compartilhando a
   * mesma autenticao do login atual da conexo. Portanto essa operao s
   * pode ser chamada enquanto a conexo estiver autenticada, caso contrrio a
   * exceo de sistema CORBA::NO_PERMISSION{NoLogin}  lanada. As informaes
   * fornecidas por essa operao devem ser passadas para a operao
   * 'loginBySharedAuth' para concluso do processo de login por autenticao
   * compartilhada. Isso deve ser feito dentro do tempo de lease definido pelo
   * administrador do barramento. Caso contrrio essas informaes se tornam
   * invlidas e no podem mais ser utilizadas para criar um login.
   * 
   * @param secret Segredo a ser fornecido na concluso do processo de login.
   * @param retries Parmetro opcional indicando o nmero de novas tentativas de
   *        busca de ofertas em caso de falhas, como o barramento estar
   *        indisponvel ou no for possvel estabelecer um login at o momento.
   *        'retries' com o valor 0 implica que a operao retorna imediatamente
   *        aps uma nica tentativa. Para tentar indefinidamente o valor de
   *        'retries' deve ser -1. Entre cada tentativa  feita uma pausa dada
   *        pelo parmetro 'interval' fornecido na criao do assistente (veja a
   *        interface 'AssistantFactory').
   * 
   * @return Objeto que representa o processo de login iniciado.
   * @throws Throwable
   */
  public LoginProcess startSharedAuth(OctetSeqHolder secret, int retries)
    throws Throwable {
    int attempt = retries;
    Throwable last;
    do {
      last = null;
      if (conn.login() != null) {
        try {
          LoginProcess process = startSharedAuthentication(secret);
          if (process != null) {
            return process;
          }
        }
        catch (Throwable e) {
          last = e;
        }
      }
    } while (shouldRetry(retries, --attempt));
    if (last != null) {
      throw last;
    }
    return null;
  }

  /**
   * Verifica se deve retentar a operao.
   * 
   * @param retries nmero de tentativas configuradas pelo usurio
   * @param attempt nmero de tentativa restantes
   * @return <code>true</code> caso deva realizar uma nova busca, e
   *         <code>false</code> caso contrrio.
   */
  private boolean shouldRetry(int retries, int attempt) {
    if (retries < 0 || attempt >= 0) {
      try {
        Thread.sleep(interval * 1000);
      }
      catch (InterruptedException e) {
        logger.log(Level.SEVERE, "'Find' foi interrompido.", e);
      }
    }
    else {
      return false;
    }
    return true;
  }

  /**
   * Encerra o funcionamento do assistente liberando todos os recursos alocados
   * por ele.
   * <p>
   * Essa operao deve ser chamada antes do assistente ser descartado, pois
   * como o assistente tem um funcionamento ativo, ele continua funcionando e
   * consumindo recursos mesmo que a aplicao no tenha mais referncias a ele.
   * Em particular, alguns dos recursos gerenciados pelo assistente so:
   * <ul>
   * <li>Login no barramento;
   * <li>Ofertas de servio registradas no barramento;
   * <li>Observadores de servio registrados no barramento;
   * <li>Threads de manuteno desses recursos no barramento;
   * <li>Conexo default no ORB sendo utilizado;
   * </ul>
   * Em particular, o processamento de requisies do ORB (e.g. atravs da
   * operao 'ORB::run()') no  gerido pelo assistente, portanto 
   * responsabilidade da aplicao iniciar e parar esse processamento (e.g.
   * atravs da operao 'ORB::shutdown()')
   */
  public void shutdown() {
    this.shutdown = true;
    threadPool.shutdownNow();
    // Aguarda o trmino da execuo das threads
    try {
      long timeout = 3 * interval;
      TimeUnit timeUnit = TimeUnit.SECONDS;
      if (!threadPool.awaitTermination(timeout, timeUnit)) {
        logger.log(Level.WARNING, String.format(
          "pool de threads no finalizou. Timeout = %s s", timeout));
      }
    }
    catch (InterruptedException e) {
      logger.log(Level.SEVERE, "pool de threads foi interrompido.", e);
    }
    try {
      conn.logout();
    }
    // bus core
    catch (ServiceFailure e) {
      logger.log(Level.SEVERE, String.format(
        "falha severa no barramento em %s:%s : %s", host, port, e.message), e);
    }
    catch (TRANSIENT e) {
      logger.log(Level.WARNING, String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port), e);
    }
    catch (COMM_FAILURE e) {
      logger.log(Level.WARNING,
        "falha de comunicao ao acessar servios ncleo do barramento", e);
    }
    catch (NO_PERMISSION e) {
      if (e.minor == NoLoginCode.value) {
        logger.log(Level.WARNING, "no h um login vlido no momento", e);
      }
      else {
        logger.log(Level.SEVERE, String.format(
          "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
      }
    }
    context.setDefaultConnection(null);
  }

  /**
   * Mtodo de obteno de dados para autenticao de login.
   * <p>
   * Mtodo a ser implementado que ser chamado quando o assistente precisar
   * autenticar uma entidade para restabelecer um login com o barramento.
   * 
   * @return Informaes para autenticao para estabelecimento de login.
   */
  public abstract AuthArgs onLoginAuthentication();

  /**
   * Mtodo responsvel por recuperar os argumentos necessrios e realizar o
   * login junto ao barramento. O mtodo retorna uma indicao se deveria
   * retentar o login , dado que ocorreu alguma falha durante o processo.
   * 
   * @return <code>true</code> caso o processo de login tenha falhado, e
   *         <code>false</code> caso seja bem sucedido.
   */
  private boolean login() {
    AuthArgs args = onLoginAuthentication();
    boolean failed = true;
    Throwable ex = null;
    try {
      if (args != null) {
        switch (args.mode) {
          case AuthByPassword:
            conn.loginByPassword(args.entity, args.password);
          case AuthByCertificate:
            conn.loginByCertificate(args.entity, args.privkey);
          case AuthBySharing:
            conn.loginBySharedAuth(args.attempt, args.secret);
        }
        failed = false;
      }
      else {
        ex =
          new NullPointerException(
            "'onLoginAuthentication' retornou argumentos de login nulos.");
      }
    }
    catch (AccessDenied e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao realizar login.", e);
    }
    catch (AlreadyLoggedIn e) {
      // ignorando o erro
      failed = false;
    }
    catch (MissingCertificate e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao realizar loginByCertificate.", e);
    }
    catch (InvalidLoginProcess e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao realizar loginBySharedAuth.", e);
    }
    // bus core
    catch (ServiceFailure e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao realizar login.", e);
    }
    catch (TRANSIENT e) {
      ex = e;
      logger.log(Level.WARNING, String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port), e);
    }
    catch (COMM_FAILURE e) {
      ex = e;
      logger.log(Level.WARNING,
        "falha de comunicao ao acessar servios ncleo do barramento", e);
    }
    catch (NO_PERMISSION e) {
      ex = e;
      if (e.minor == NoLoginCode.value) {
        logger.log(Level.WARNING, "no h um login vlido no momento", e);
      }
      else {
        logger.log(Level.SEVERE, String.format(
          "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
      }
    }
    catch (Throwable e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro inesperado!", e);
    }
    finally {
      if (failed) {
        try {
          callback.onLoginFailure(this, ex);
        }
        catch (Throwable e) {
          logger.log(Level.SEVERE, "Erro inesperado ao chamar callback!", e);
        }
      }
    }
    return failed;
  }

  /**
   * Mtodo responsvel por buscar por servios que atendam as propriedades
   * especificadas.
   * 
   * @param props as propriedades de servio que se deseja buscar
   * 
   * @return as ofertas de servios encontradas, ou <code>null</code> caso algum
   *         erro tenha ocorrido.
   * @throws Throwable
   */
  private ServiceOfferDesc[] find(ServiceProperty[] props) throws Throwable {
    boolean failed = true;
    Throwable ex = null;
    ServiceOfferDesc[] offerDescs = null;
    try {
      OfferRegistry offerRegistry = context.getOfferRegistry();
      offerDescs = offerRegistry.findServices(props);
      failed = false;
    }
    // bus core
    catch (ServiceFailure e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao buscar ofertas.", e);
    }
    catch (TRANSIENT e) {
      ex = e;
      logger.log(Level.WARNING, String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port), e);
    }
    catch (COMM_FAILURE e) {
      ex = e;
      logger.log(Level.WARNING,
        "falha de comunicao ao acessar servios ncleo do barramento", e);
    }
    catch (NO_PERMISSION e) {
      ex = e;
      if (e.minor == NoLoginCode.value) {
        logger.log(Level.WARNING, "no h um login vlido no momento", e);
      }
      else {
        logger.log(Level.SEVERE, String.format(
          "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
      }
    }
    catch (Throwable e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro inesperado!", e);
    }
    finally {
      if (failed) {
        try {
          callback.onFindFailure(this, ex);
        }
        catch (Throwable e) {
          logger.log(Level.SEVERE, "Erro inesperado ao chamar callback!", e);
        }
        throw ex;
      }
    }
    return offerDescs;
  }

  /**
   * Mtodo responsvel por buscar todas as ofertas de servio publicadas no
   * barramento.
   * 
   * @return as ofertas de servios encontradas, ou <code>null</code> caso algum
   *         erro tenha ocorrido.
   * @throws Throwable
   */
  private ServiceOfferDesc[] getAll() throws Throwable {
    boolean failed = true;
    Throwable ex = null;
    ServiceOfferDesc[] offerDescs = null;
    try {
      OfferRegistry offerRegistry = context.getOfferRegistry();
      offerDescs = offerRegistry.getAllServices();
      failed = false;
    }
    // bus core
    catch (ServiceFailure e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro ao recuperar ofertas.", e);
    }
    catch (TRANSIENT e) {
      ex = e;
      logger.log(Level.WARNING, String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port), e);
    }
    catch (COMM_FAILURE e) {
      ex = e;
      logger.log(Level.WARNING,
        "falha de comunicao ao acessar servios ncleo do barramento", e);
    }
    catch (NO_PERMISSION e) {
      ex = e;
      if (e.minor == NoLoginCode.value) {
        logger.log(Level.WARNING, "no h um login vlido no momento", e);
      }
      else {
        logger.log(Level.SEVERE, String.format(
          "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
      }
    }
    catch (Throwable e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro inesperado!", e);
    }
    finally {
      if (failed) {
        try {
          callback.onFindFailure(this, ex);
        }
        catch (Throwable e) {
          logger.log(Level.SEVERE, "Erro inesperado ao chamar callback!", e);
        }
        throw ex;
      }
    }
    return offerDescs;
  }

  /**
   * Mtodo responsvel por iniciar o processo de compartilhamento de
   * autenticao.
   * 
   * @param secret segredo a ser fornecido na concluso do porcesso de login.
   * @return Objeto que representa o processo de login iniciado, ou
   *         <code>null</code> caso algum erro tenha ocorrido.
   * @throws Throwable
   */
  private LoginProcess startSharedAuthentication(OctetSeqHolder secret)
    throws Throwable {
    boolean failed = true;
    Throwable ex = null;
    LoginProcess attempt = null;
    try {
      attempt = this.conn.startSharedAuth(secret);
      failed = false;
    }
    // bus core
    catch (ServiceFailure e) {
      ex = e;
      logger.log(Level.SEVERE,
        "Erro ao iniciar processo de autenticao compartilhada.", e);
    }
    catch (TRANSIENT e) {
      ex = e;
      logger.log(Level.WARNING, String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port), e);
    }
    catch (COMM_FAILURE e) {
      ex = e;
      logger.log(Level.WARNING,
        "falha de comunicao ao acessar servios ncleo do barramento", e);
    }
    catch (NO_PERMISSION e) {
      ex = e;
      if (e.minor == NoLoginCode.value) {
        logger.log(Level.WARNING, "no h um login vlido no momento", e);
      }
      else {
        logger.log(Level.SEVERE, String.format(
          "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
      }
    }
    catch (Throwable e) {
      ex = e;
      logger.log(Level.SEVERE, "Erro inesperado!", e);
    }
    finally {
      if (failed) {
        try {
          callback.onStartSharedAuthFailure(this, ex);
        }
        catch (Throwable e) {
          logger.log(Level.SEVERE, "Erro inesperado ao chamar callback!", e);
        }
        throw ex;
      }
    }
    return attempt;
  }

  /**
   * Classe interna do Assistente que representa uma oferta a se manter
   * registrada no barramento.
   * 
   * @author Tecgraf
   */
  private class Offer {

    /** O assistente */
    Assistant assist;
    /** O Componente a ser registrado */
    IComponent component;
    /** Propriedades a serem cadastradas na oferta */
    ServiceProperty[] properties;
    /** Referncia para a descrio da oferta */
    AtomicReference<ServiceOfferDesc> offer =
      new AtomicReference<ServiceOfferDesc>();
    /** Lock da oferta */
    Object lock = new Object();
    /** Varivel condicional que informa que um evento de INVALID LOGIN ocorreu */
    AtomicBoolean event = new AtomicBoolean(true);

    /**
     * Construtor.
     * 
     * @param assist o assistente
     * @param component o componente a ser ofertado
     * @param properties as propriedades com as quais a oferta deve se cadastrar
     */
    public Offer(Assistant assist, IComponent component,
      ServiceProperty[] properties) {
      this.assist = assist;
      this.component = component;
      this.properties = Arrays.copyOf(properties, properties.length);
    }

    /**
     * Recupera o login da conexo que registrou a oferta.
     * 
     * @return o login da conexo que registrou a oferta, ou <code>null</code>
     *         caso no tenha sido registrada nenhuma vez.
     */
    public String loginId() {
      ServiceOfferDesc desc = offer.get();
      if (desc != null) {
        for (ServiceProperty prop : desc.properties) {
          if (prop.name.equals("openbus.offer.login")) {
            return prop.value;
          }
        }
      }
      return null;
    }

    /**
     * Mtodo responsvel por registrar a oferta de servio no barramento. O
     * mtodo retorna uma indicao se deveria retentar o login , dado que
     * ocorreu alguma falha durante o processo
     * 
     * @return <code>true</code> caso o registro tenha falhado, e
     *         <code>false</code> caso seja bem sucedido.
     */
    public boolean registryOffer() {
      boolean failed = true;
      Throwable ex = null;
      try {
        OfferRegistry offerRegistry = assist.context.getOfferRegistry();
        ServiceOffer theOffer =
          offerRegistry.registerService(component, properties);
        offer.set(theOffer.describe());
        failed = false;
      }
      // register
      catch (UnauthorizedFacets e) {
        StringBuffer interfaces = new StringBuffer();
        for (String facet : e.facets) {
          interfaces.append("\n  - ");
          interfaces.append(facet);
        }
        ex = e;
        logger
          .log(
            Level.WARNING,
            String
              .format(
                "a entidade no foi autorizada pelo administrador do barramento a ofertar os servios: %s",
                interfaces.toString()), e);
      }
      catch (InvalidService e) {
        ex = e;
        logger.log(Level.WARNING,
          "o servio ofertado apresentou alguma falha durante o registro.", e);
      }
      catch (InvalidProperties e) {
        StringBuffer props = new StringBuffer();
        for (ServiceProperty prop : e.properties) {
          props.append("\n  - ");
          props.append(String.format("name = %s, value = %s", prop.name,
            prop.value));
        }
        ex = e;
        logger.log(Level.WARNING, String.format(
          "tentativa de registrar servio com propriedades invlidas: %s",
          props.toString()), e);
      }
      // bus core
      catch (ServiceFailure e) {
        ex = e;
        logger.log(Level.SEVERE, "Erro ao realizar login.", e);
      }
      catch (TRANSIENT e) {
        ex = e;
        logger.log(Level.WARNING, String.format(
          "o barramento em %s:%s esta inacessvel no momento", assist.host,
          assist.port), e);
      }
      catch (COMM_FAILURE e) {
        ex = e;
        logger.log(Level.WARNING,
          "falha de comunicao ao acessar servios ncleo do barramento", e);
      }
      catch (NO_PERMISSION e) {
        ex = e;
        if (e.minor == NoLoginCode.value) {
          logger.log(Level.WARNING, "no h um login vlido no momento", e);
        }
        else {
          logger.log(Level.SEVERE, String.format(
            "erro de NO_PERMISSION no esperado: minor_code = %s", e.minor), e);
        }
      }
      catch (Throwable e) {
        ex = e;
        logger.log(Level.SEVERE, "Erro inesperado!", e);
      }
      finally {
        if (failed) {
          try {
            assist.callback
              .onRegisterFailure(assist, component, properties, ex);
          }
          catch (Throwable e) {
            logger.log(Level.SEVERE, "Erro inesperado ao chamar callback!", e);
          }
        }
      }
      return failed;
    }

    /**
     * Marca a oferta como invlida para que o registro seja refeito.
     */
    public void reset() {
      synchronized (this.lock) {
        this.event.set(true);
        this.lock.notifyAll();
        logger.fine("Resetando oferta.");
      }
    }
  }

  /**
   * Callback do assistente a ser chamado quando ocorrer uma exceo
   * {@link NO_PERMISSION} com minor igual a {@link InvalidLoginCode}
   * 
   * @author Tecgraf
   */
  private class OnInvalidLogin implements InvalidLoginCallback {

    /**
     * Refaz o login e dispara threads para registrar as ofertas novamente.
     * {@inheritDoc}
     */
    @Override
    public void invalidLogin(Connection conn, LoginInfo login) {
      logger.fine("Iniciando callback 'OnInvalidLogin");
      DoLogin doLogin = new DoLogin(Assistant.this);
      doLogin.run();
      synchronized (Assistant.this.offers) {
        for (Offer aOffer : Assistant.this.offers) {
          aOffer.reset();
        }
      }
      logger.fine("Finalizando callback 'OnInvalidLogin");
    }

  }

  /**
   * Tarefa que executa o processo de login junto ao barramento.
   * 
   * @author Tecgraf
   */
  private class DoLogin implements Runnable {

    /** Assistente em uso */
    Assistant assist;

    /**
     * Construtor
     * 
     * @param assist o assistente em uso.
     */
    public DoLogin(Assistant assist) {
      this.assist = assist;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      boolean retry = true;
      logger.fine("Iniciando tarefa 'DoLogin'.");
      if (assist.conn.login() != null) {
        // j possui um login vlido
        retry = false;
      }
      while (retry && !assist.shutdown) {
        retry = assist.login();
        if (retry) {
          try {
            Thread.sleep(assist.interval * 1000);
          }
          catch (InterruptedException e) {
            logger.fine("Thread 'DoLogin' foi interrompida.");
          }
        }
      }
      logger.fine("Finalizando tarefa 'DoLogin'.");
    }
  }

  /**
   * Tarefa que executa o processo de registro de oferta junto ao barramento.
   * 
   * @author Tecgraf
   */
  private class DoRegister implements Runnable {

    /** Assistente em uso */
    Assistant assist;
    /** Oferta de servio a ser registrada */
    private Offer offer;

    /**
     * Construtor
     * 
     * @param assist o assistente em uso.
     * @param offer a oferta a ser registrada.
     */
    public DoRegister(Assistant assist, Offer offer) {
      this.assist = assist;
      this.offer = offer;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      logger.fine("Iniciando tarefa 'DoRegister'.");
      while (!assist.shutdown) {
        boolean retry = true;
        // espera por notificao de evento
        synchronized (offer.lock) {
          while (!offer.event.get() && !assist.shutdown) {
            try {
              offer.lock.wait();
            }
            catch (InterruptedException e) {
              logger.fine("Thread 'DoRegister' foi interrompida.");
              break;
            }
          }
          offer.event.set(false);
        }
        // tratando 1 evento
        while (retry && !assist.shutdown) {
          logger.fine("Thread 'DoRegister' tratando evento");
          LoginInfo login = assist.conn.login();
          if (login != null) {
            if (!login.id.equals(offer.loginId())) {
              retry = offer.registryOffer();
            }
            else {
              retry = false;
            }
          }
          if (retry) {
            try {
              Thread.sleep(assist.interval * 1000);
            }
            catch (InterruptedException e) {
              logger.warning("Thread 'DoRegister' foi interrompida.");
              break;
            }
          }
          else {
            logger.fine("Completou atendimento de evento 'DoRegister'.");
          }
        }
      }
      logger.fine("Finalizando tarefa 'DoRegister'.");
    }
  }

  /**
   * Implementao padro da callback de falhas de execuo do assistente.
   * 
   * @author Tecgraf
   */
  private class DefaultFailureCallback implements OnFailureCallback {

    /**
     * {@inheritDoc}
     */
    @Override
    public void onLoginFailure(Assistant assistant, Throwable exception) {
      // do nothing
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onRegisterFailure(Assistant assistant, IComponent component,
      ServiceProperty[] properties, Throwable except) {
      // do nothing
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onFindFailure(Assistant assistant, Throwable exception) {
      // do nothing
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onStartSharedAuthFailure(Assistant assistant, Throwable except) {
      // do nothing
    }

  }
}
