package tecgraf.openbus.assistant;

import tecgraf.openbus.PrivateKey;
import tecgraf.openbus.core.v2_0.services.access_control.LoginProcess;

/**
 * Informaes de autenticao de entidades.
 * 
 * @author Tecgraf
 */
public class AuthArgs {

  /**
   * Enumerao dos tipos de autenticao suportados pelo assistente.
   * 
   * @author Tecgraf
   */
  enum AuthMode {
    /** Autenticao por senha */
    AuthByPassword,
    /** Autenticao por certificado */
    AuthByCertificate,
    /** Autenticao compartilhada */
    AuthBySharing;
  }

  /** Modo de autenticao */
  AuthMode mode;
  /** Entidade */
  String entity;
  /** Senha */
  byte[] password;
  /** Chave privada */
  PrivateKey privkey;
  /** Processo de compartilhamento de login */
  LoginProcess attempt;
  /** Segredo do compartilhamento de login */
  byte[] secret;

  /**
   * Construtor para realizar autenticao por senha
   * 
   * @param entity Identificador da entidade a ser autenticada.
   * @param password Senha de autenticao no barramento da entidade.
   */
  public AuthArgs(String entity, byte[] password) {
    if (entity == null || password == null) {
      throw new IllegalArgumentException(
        "Entidade e senha devem ser diferentes de nulo.");
    }
    this.entity = entity;
    this.password = password;
    this.mode = AuthMode.AuthByPassword;
  }

  /**
   * Construtor para realizar autenticao por senha
   * 
   * @param entity Identificador da entidade a ser autenticada.
   * @param privkey Chave privada correspondente ao certificado registrado a ser
   *        utilizada na autenticao.
   */
  public AuthArgs(String entity, PrivateKey privkey) {
    if (entity == null || privkey == null) {
      throw new IllegalArgumentException(
        "Entidade e chave privada devem ser diferentes de nulo.");
    }
    this.entity = entity;
    this.privkey = privkey;
    this.mode = AuthMode.AuthByCertificate;
  }

  /**
   * Construtor para realizar autenticao compartilhada
   * 
   * @param attempt Objeto que represeta o processo de login iniciado.
   * @param secret Segredo a ser fornecido na concluso do processo de login.
   */
  public AuthArgs(LoginProcess attempt, byte[] secret) {
    if (attempt == null || secret == null) {
      throw new IllegalArgumentException(
        "Processo de login e segredo devem ser diferentes de nulo.");
    }
    this.attempt = attempt;
    this.secret = secret;
    this.mode = AuthMode.AuthBySharing;
  }
}
