package tecgraf.openbus.core;

import java.util.concurrent.atomic.AtomicInteger;


/**
 * Classe que representa a parte comum entre a sesso do lado servidor e
 * cliente.
 * 
 * @author Tecgraf
 */
abstract class Session {

  /**
   * Identificador da sesso.
   */
  final private int session;
  /**
   * O segredo compartilhado na sesso
   */
  final protected byte[] secret;

  /**
   * Construtor.
   * 
   * @param session identificador da sesso
   * @param secret o segredo.
   */
  public Session(int session, byte[] secret) {
    this.session = session;
    this.secret = secret;
  }

  /**
   * Recupera o identificador da sesso.
   * 
   * @return o identificador da sesso.
   */
  public int getSession() {
    return this.session;
  }

  /**
   * Recupera o segredo.
   * 
   * @return o segredo.
   */
  public byte[] getSecret() {
    return this.secret;
  }

  /**
   * Representa a sesso do lado servidor.
   * 
   * @author Tecgraf
   */
  static class ServerSideSession extends Session {
    /**
     * Histrico dos tickets.
     */
    private TicketsHistory ticket;
    /**
     * Originador da comunicao.
     */
    final private String caller;

    /**
     * Construtor.
     * 
     * @param session identificador da sesso
     * @param secret o segredo.
     * @param caller originador da comunicao
     */
    public ServerSideSession(int session, byte[] secret, String caller) {
      super(session, secret);
      this.ticket = new TicketsHistory();
      this.caller = caller;
    }

    /**
     * Verifica se o ticket  vlido e marca com utilizado caso seja vlido.
     * 
     * @param id o ticket a ser utilizado.
     * @return <code>true</code> caso o ticket era vlido e foi marcado, e
     *         <code>false</code> caso o ticket no fosse vlido.
     */
    public boolean checkTicket(int id) {
      return this.ticket.check(id);
    }

    /**
     * Recupera a informao do originador da comunicao
     * 
     * @return o originador da comunicao.
     */
    public String getCaller() {
      return this.caller;
    }

  }

  /**
   * Repesenta a sesso do lado cliente.
   * 
   * @author Tecgraf
   */
  static class ClientSideSession extends Session {
    /**
     * Valor do ltimo ticket utilizado
     */
    private AtomicInteger ticket;
    /**
     * Alvo da comunicao.
     */
    final private String callee;

    /**
     * Construtor.
     * 
     * @param session identificador da sesso
     * @param secret o segredo.
     * @param callee alvo da comunicao
     */
    public ClientSideSession(int session, byte[] secret, String callee) {
      super(session, secret);
      this.ticket = new AtomicInteger(-1);
      this.callee = callee;
    }

    /**
     * Recupera o valor do prximo ticket.
     * 
     * @return o valor do prximo ticket.
     */
    public int nextTicket() {
      return this.ticket.incrementAndGet();
    }

    /**
     * Recupera a informao do alvo da comunicao
     * 
     * @return o alvo da comunicao.
     */
    public String getCallee() {
      return this.callee;
    }

  }
}
