package demo;

import java.util.HashMap;
import java.util.Map.Entry;

import org.omg.CORBA.COMM_FAILURE;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;
import org.omg.CORBA.TRANSIENT;
import org.omg.CORBA.ORBPackage.InvalidName;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;
import org.omg.PortableServer.POAManagerPackage.AdapterInactive;

import scs.core.ComponentContext;
import scs.core.ComponentId;
import scs.core.exception.SCSException;
import tecgraf.openbus.Connection;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.core.ORBInitializer;
import tecgraf.openbus.core.OpenBusPrivateKey;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.MissingCertificate;
import tecgraf.openbus.core.v2_0.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidProperties;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidService;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;
import tecgraf.openbus.core.v2_0.services.offer_registry.UnauthorizedFacets;
import tecgraf.openbus.demo.util.Utils;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import demo.GreetingsImpl.Language;
import demo.GreetingsImpl.Period;

/**
 * Parte servidora do demo Hello
 * 
 * @author Tecgraf
 */
public final class GreetingsServer {

  private static String host;
  private static int port;
  private static String entity;
  private static OpenBusPrivateKey privateKey;

  /**
   * Funo principal.
   * 
   * @param args argumentos.
   * @throws InvalidName
   * @throws AdapterInactive
   * @throws SCSException
   * @throws AlreadyLoggedIn
   * @throws ServiceFailure
   */
  public static void main(String[] args) throws InvalidName, AdapterInactive,
    SCSException, AlreadyLoggedIn, ServiceFailure {
    // verificando parametros de entrada
    if (args.length < 4) {
      System.out.println(String.format(Utils.serverUsage, "", ""));
      System.exit(1);
      return;
    }
    // - host
    host = args[0];
    // - porta
    try {
      port = Integer.parseInt(args[1]);
    }
    catch (NumberFormatException e) {
      System.out.println(Utils.port);
      System.exit(1);
      return;
    }
    // - entidade
    entity = args[2];
    // - chave privada
    String privateKeyFile = args[3];
    try {
      privateKey = OpenBusPrivateKey.createPrivateKeyFromFile(privateKeyFile);
    }
    catch (Exception e) {
      System.out.println(Utils.keypath);
      e.printStackTrace();
      System.exit(1);
      return;
    }

    // inicializando e configurando o ORB
    final ORB orb = ORBInitializer.initORB();
    // - disparando a thread para que o ORB atenda requisies
    Thread run = new Thread() {
      @Override
      public void run() {
        orb.run();
      }
    };
    run.start();
    // - criando thread para parar e destruir o ORB ao fim da execuo do processo 
    Thread shutdown = new Thread() {
      @Override
      public void run() {
        orb.shutdown(true);
        orb.destroy();
      }
    };
    Runtime.getRuntime().addShutdownHook(shutdown);

    // recuperando o gerente de contexto de chamadas a barramentos 
    OpenBusContext context =
      (OpenBusContext) orb.resolve_initial_references("OpenBusContext");

    // criando o servio a ser ofertado
    // - ativando o POA
    POA poa = POAHelper.narrow(orb.resolve_initial_references("RootPOA"));
    poa.the_POAManager().activate();
    // - construindo os componentes
    HashMap<Language, ComponentContext> components =
      new HashMap<GreetingsImpl.Language, ComponentContext>();
    for (Language language : Language.values()) {
      ComponentId id =
        new ComponentId(language.name() + Greetings.class.getSimpleName(),
          (byte) 1, (byte) 0, (byte) 0, "java");
      ComponentContext component = new ComponentContext(orb, poa, id);
      for (Period period : Period.values()) {
        component.addFacet("Good" + period.name(), GreetingsHelper.id(),
          new GreetingsImpl(context, language, period));
      }
      components.put(language, component);
    }

    // conectando ao barramento.
    Connection conn = context.createConnection(host, port);
    context.setDefaultConnection(conn);

    // autentica-se no barramento
    boolean failed = true;
    try {
      conn.loginByCertificate(entity, privateKey);
      // registrando servio no barramento
      ServiceProperty[] serviceProperties = new ServiceProperty[2];
      serviceProperties[0] =
        new ServiceProperty("offer.domain", "Demo Greetings");
      for (Entry<Language, ComponentContext> entry : components.entrySet()) {
        serviceProperties[1] =
          new ServiceProperty("greetings.language", entry.getKey().name());
        context.getOfferRegistry().registerService(
          entry.getValue().getIComponent(), serviceProperties);
      }
      failed = false;
    }
    // login by certificate
    catch (AccessDenied e) {
      System.err.println(String.format(
        "a chave em '%s' no corresponde ao certificado da entidade '%s'",
        privateKeyFile, entity));
    }
    catch (MissingCertificate e) {
      System.err.println(String.format(
        "a entidade %s no possui um certificado registrado", entity));
    }
    // register
    catch (UnauthorizedFacets e) {
      StringBuffer interfaces = new StringBuffer();
      for (String facet : e.facets) {
        interfaces.append("\n  - ");
        interfaces.append(facet);
      }
      System.err
        .println(String
          .format(
            "a entidade '%s' no foi autorizada pelo administrador do barramento a ofertar os servios: %s",
            entity, interfaces.toString()));
    }
    catch (InvalidService e) {
      System.err
        .println("o servio ofertado apresentou alguma falha durante o registro.");
    }
    catch (InvalidProperties e) {
      StringBuffer props = new StringBuffer();
      for (ServiceProperty prop : e.properties) {
        props.append("\n  - ");
        props.append(String.format("name = %s, value = %s", prop.name,
          prop.value));
      }
      System.err.println(String.format(
        "tentativa de registrar servio com propriedades invlidas: %s", props
          .toString()));
    }
    // bus core
    catch (ServiceFailure e) {
      System.err.println(String.format(
        "falha severa no barramento em %s:%s : %s", host, port, e.message));
    }
    catch (TRANSIENT e) {
      failed = true;
      System.err.println(String.format(
        "o barramento em %s:%s esta inacessvel no momento", host, port));
    }
    catch (COMM_FAILURE e) {
      System.err
        .println("falha de comunicao ao acessar servios ncleo do barramento");
    }
    catch (NO_PERMISSION e) {
      if (e.minor == NoLoginCode.value) {
        System.err.println(String.format(
          "no h um login de '%s' vlido no momento", entity));
      }
    }
    finally {
      if (failed) {
        context.getCurrentConnection().logout();
        System.exit(1);
      }
    }

  }
}
