package csbase.client.algorithms.commands.newview;

import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.client.kernel.ClientException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.ProgressData;
import csbase.logic.algorithms.AlgorithmConfigurator;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Viso de um n de um fluxo.
 */
class NodeCommandView extends AbstractCommandView {

  /**
   * N do fluxo a ser visualizado.
   */
  private int nodeId;

  /**
   * Construtor.
   * 
   * @param nodeId n do fluxo a ser visualizado.
   * @param cmd o comando sendo visualizado (No aceita {@code null}).
   * @param configurator o configurador do algoritmo executado pelo comando (No
   *        aceita {@code null}).
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  public NodeCommandView(int nodeId, CommandInfo cmd,
    AlgorithmConfigurator configurator) throws ClientException {
    super(CommandViewType.NODE, cmd, configurator);
    this.nodeId = nodeId;
    this.idForLogPattern = String.valueOf(nodeId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getFinalizationTypeDescription() {
    CommandFinalizationInfo finalizationInfo = command.getFinalizationInfo();
    if (finalizationInfo.getInfoType() == FinalizationInfoType.SIMPLE) {
      // Se s houver informao de finalizao do comando como um todo.
      return super.getFinalizationTypeDescription();
    }
    else {
      // Se houver informao de finalizao especfica para o n
      ExtendedCommandFinalizationInfo flowfinalizationInfo =
        (ExtendedCommandFinalizationInfo) finalizationInfo;
      CommandFinalizationInfo nodeFinalizationInfo =
        flowfinalizationInfo.getFinalizationInfoForNode(nodeId);
      String description;
      switch (nodeFinalizationInfo.getFinalizationType()) {
        case EXECUTION_ERROR:
          description = getFinalizationTypeDescription(
            CommandFinalizationType.EXECUTION_ERROR,
            nodeFinalizationInfo.getExitCode());
          break;
        case SUCCESS:
          description =
            getFinalizationTypeDescription(CommandFinalizationType.SUCCESS,
              null);
          break;
        case NO_EXIT_CODE:
          description =
            getFinalizationTypeDescription(CommandFinalizationType.NO_EXIT_CODE,
              null);
          break;
        default:
          description =
            getFinalizationTypeDescription(CommandFinalizationType.END, null);
      }
      if (nodeFinalizationInfo.hasWarnings()) {
        description +=
          " " + LNG.get("AbstractAlgorithmCommandView.status.warning");
      }
      return description;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProgressData getCommandProgress() {
    Map<Integer, ProgressData> progressDataMap = command.getProgressDataMap();
    return progressDataMap.get(nodeId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<LogTabConfiguration, Set<ClientProjectFile>> getMatchingLogFiles(
    List<LogTabConfiguration> tabConfiguration) {
    String[] logsDir = command.getPersistencyPath();

    MatchLogTabConfigurationsTask task =
      new MatchLogTabConfigurationsTask(logsDir, tabConfiguration);
    String message =
      LNG
        .get(NodeCommandView.class.getSimpleName() + ".message.task.load.logs");
    if (task.execute(owner, getTitle(), message)) {
      return task.getResult();
    }
    else {
      String errorMsg =
        LNG.get(
          NodeCommandView.class.getSimpleName() + ".error.task.load.logs",
          new Object[] { command.getId() });
      StandardDialogs.showErrorDialog(this.mainPanel.getParent(), getTitle(),
        errorMsg);
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTitle() {
    return LNG.get(
      NodeCommandView.class.getSimpleName() + ".window.title",
      new Object[] { configurator.getAlgorithmName(),
          configurator.getAlgorithmVersionId() });
  }

}
