/*
 * $Id:$
 */
package csbase.client.algorithms.view.simple;

import java.awt.GridBagLayout;
import java.awt.Window;
import java.rmi.RemoteException;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;

import csbase.client.algorithms.parameters.ParameterGroupView;
import csbase.client.algorithms.parameters.ParameterGroupViewListener;
import csbase.client.algorithms.parameters.ParameterView;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.algorithms.parameters.ParameterViewListener;
import csbase.client.algorithms.parameters.SimpleParameterView;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.algorithms.view.AbstractAlgorithmConfiguratorPanel;
import csbase.client.kernel.ClientException;
import csbase.exception.algorithms.FormulaEvaluationException;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.ValidationExpression;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.panel.ExpandablePanel;
import tecgraf.javautils.gui.panel.ExpandablePanelGroup;

/**
 * Painel representando a viso dos parmetros de um configurador simples.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SimpleAlgorithmConfiguratorPanel extends
  AbstractAlgorithmConfiguratorPanel<SimpleAlgorithmConfigurator> {

  /**
   * Lista de vises de grupo
   */
  private List<ParameterGroupView> groupViews;

  /**
   * @param owner a janela que detm este painel (No aceita {@code null} ).
   * @param configurator o configurador.
   * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
   *        {@link Mode#REPORT}.
   *
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public SimpleAlgorithmConfiguratorPanel(Window owner,
    SimpleAlgorithmConfigurator configurator, ParameterView.Mode mode)
    throws ClientException {
    super(owner, configurator, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AbstractParametersPanel createParametersConfigurationPanel() {
    return new SimpleParametersConfigurationPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AbstractParametersPanel createParametersReportPanel() {
    return new SimpleParametersReportPanel();
  }

  /**
   * Atualiza a viso de todos os grupos do painel.
   */
  public void updateViews() {
    for (ParameterGroupView groupView : groupViews) {
      for (SimpleParameterView<?> parameterView : groupView
        .getParameterViews()) {
        parameterView.updateViewContents();
      }
    }
  }

  /**
   * Adio de listaner em todos os grupos do painel.
   *
   * @param listener o listener
   */
  public void addParameterGroupViewListener(ParameterGroupViewListener listener) {
    for (ParameterGroupView aGroupView : groupViews) {
      aGroupView.addParameterGroupViewListener(listener);
    }
  }

  /**
   * Painel interno que detm a viso dos parmetros.
   */
  private abstract class AbstractSimpleParametersPanel extends
    AbstractParametersPanel {

    /**
     * Painel principal, que contm os parmetros.
     */
    private JScrollPane mainPanel;

    /**
     * Construtor.
     *
     * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
     *        {@link Mode#REPORT}.
     */
    protected AbstractSimpleParametersPanel(final Mode mode) {
      super();
      executeBeforeInitialization();
      initialize(mode);
      executeAfterInitialization();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean highlightValidationResult(ViewValidationResult result) {
      ViewValidator element = result.getElement();
      boolean highlighted = false;

      if (element.equals(this) && result.getChildren() != null) {
        for (Validation childResult : result.getChildren()) {
          if (childResult instanceof ViewValidationResult) {
            ViewValidationResult viewChildResult =
              (ViewValidationResult) childResult;
            highlighted |= viewChildResult.getElement()
              .highlightValidationResult(viewChildResult);
          }
        }
      }
      if (highlighted && !result.isWellSucceeded()) {
        return true;
      }

      return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ViewValidationResult validate(ValidationMode mode)
      throws RemoteException {

      ViewValidationResult validation = new ViewValidationResult(this);

      for (ParameterGroupView groupView : groupViews) {
        validation.addChild(groupView.validate(mode));
      }

      // Se tiver algum parmetro errado no faz validao inter-parmetro
      if (!validation.isWellSucceeded()) {
        return validation;
      }

      SimpleAlgorithmConfigurator configurator = getConfigurator();
      for (ValidationExpression expression : configurator.getExpressions()) {
        try {
          if (!expression.evaluate()) {
            List<SimpleParameterView<?>> parameterViews =
              new LinkedList<SimpleParameterView<?>>();
            for (SimpleParameter<?> parameter : expression.getParameters()) {
              SimpleParameterView<?> parameterView = getParameterView(parameter
                .getName());
              parameterViews.add(parameterView);
            }
            for (SimpleParameterView<?> parameterView : parameterViews) {
              parameterView.highlightError();
            }
            for (SimpleParameterView<?> parameterView : parameterViews) {
              parameterView.clearError();
            }
            LocalizedMessage message = new LocalizedMessage(
              SimpleAlgorithmConfiguratorPanel.class, "expression.error",
              new Object[] { expression.getErrorMessage() });
            validation.addChild(new ViewValidationResult(new ValidationError(
              message), this));
          }
        }
        catch (FormulaEvaluationException e) {
          LocalizedMessage message = new LocalizedMessage(
            SimpleAlgorithmConfiguratorPanel.class, "formula.error",
            new Object[] { e.getLocalizedMessage() });
          validation.addChild(new ViewValidationResult(new ValidationError(
            message), this));
        }
      }
      return validation;
    }

    /**
     * Mtodo a ser executado logo antes da inicializao do painel.
     */
    protected void executeBeforeInitialization() {
      /*
       * Subclasses deste painel podem sobrescrever esse mtodo para fazer
       * configuraes pr-inicializao.
       */
    }

    /**
     * Mtodo a ser executado logo depois da inicializao do painel.
     */
    protected void executeAfterInitialization() {
      /*
       * Subclasses deste painel podem sobrescrever esse mtodo para fazer
       * configuraes ps-inicializao.
       */
    }

    /**
     * Inicializa o painel.
     *
     * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
     *        {@link Mode#REPORT}.
     */
    private void initialize(final Mode mode) {
      groupViews = createGroupsView(mode);
      mainPanel = new JScrollPane();
      setLayout(new GridBagLayout());
      populateComponent(mainPanel, mode);
      add(mainPanel, new GBC(0, 0).both());
    }

    /**
     * Liga os componentes ao componente principal.
     *
     * @param scrollPane painel principal.
     * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
     *        {@link Mode#REPORT}.
     */
    private void populateComponent(JScrollPane scrollPane, final Mode mode) {
      /*
       * Guarda a posio atual das barras de rolagem para poder manter seu
       * valor quando o painel for refeito.
       */
      int verticalScrollValue = 0;
      JScrollBar verticalScrollBar = scrollPane.getVerticalScrollBar();
      if (verticalScrollBar != null) {
        verticalScrollValue = verticalScrollBar.getValue();
      }
      JScrollBar horizontalScrollBar = scrollPane.getHorizontalScrollBar();
      int horizontalScrollValue = 0;
      if (horizontalScrollBar != null) {
        horizontalScrollValue = horizontalScrollBar.getValue();
      }

      JPanel groupViewsPanel = new JPanel(new GridBagLayout());
      GBC gbc =
        new GBC(1, 1).gridwidth(1).gridheight(1).weightx(1).insets(2, 2, 2, 2)
          .northwest();
      ExpandablePanelGroup panelGroup = new ExpandablePanelGroup();
      int gridy = gbc.gridy + 1;

      boolean fillsVerticalSpace = false;
      for (ParameterGroupView groupView : groupViews) {
        JComponent aGroupViewPanel = groupView.getPanel();
        if (aGroupViewPanel != null) {
          groupView.populatePanel();
          if (aGroupViewPanel instanceof ExpandablePanel) {
            ExpandablePanel expandablePanel = (ExpandablePanel) aGroupViewPanel;
            if (groupView.fillVerticalSpace()) {
              GBC newGBC = new GBC(gbc).horizontal().weighty(0).gridy(gridy);
              GridBagLayout layout = new GridBagLayout();
              layout.setConstraints(expandablePanel, newGBC);
              expandablePanel.setLayout(layout);
            }
            panelGroup.add(expandablePanel);
          }
          GBC newGBC = new GBC(gbc).gridy(gridy++);
          if (groupView.fillVerticalSpace()) {
            newGBC.both();
          }
          else {
            newGBC.horizontal();
          }
          groupViewsPanel.add(aGroupViewPanel, newGBC);
        }
      }
      /*
       * Se nenhum dos grupos de parmetros contm componentes que utilizam o
       * espao vertical excedente, criamos um painel extra para receber esse
       * espao. Dessa forma, evitamos que os demais painis se movam
       * verticalmente quando  feito o redimencionamento da janela.
       */
      GBC newGBC = new GBC(gbc).gridy(gridy++);
      if (!fillsVerticalSpace) {
        newGBC.both();
        groupViewsPanel.add(new JPanel(), newGBC);
      }
      scrollPane.setViewportView(groupViewsPanel);
      if (verticalScrollBar != null) {
        verticalScrollBar.setValue(verticalScrollValue);
      }
      if (horizontalScrollBar != null) {
        horizontalScrollBar.setValue(horizontalScrollValue);
      }
      scrollPane.revalidate();
    }

    /**
     * Cria a viso dos grupos de parmetro dessa algoritmo.
     *
     * @param mode modo.
     * @return A lista de vises criadas.
     */
    private List<ParameterGroupView> createGroupsView(
      final ParameterView.Mode mode) {

      List<ParameterGroupView> views = new LinkedList<ParameterGroupView>();
      for (ParameterGroup group : getConfigurator().getGroups()) {
        ParameterGroupView parameterGroupView = new ParameterGroupView(
          SimpleAlgorithmConfiguratorPanel.this, group, mode);
        views.add(parameterGroupView);
        parameterGroupView.addParameterViewListener(
          new ParameterViewListener() {
            @Override
            public void visibilityWasChanged(ParameterView<?> view) {
              populateComponent(mainPanel, mode);
            }
          });
        parameterGroupView.addParameterGroupViewListener(
          new ParameterGroupViewListener() {
            @Override
            public void childParameterVisibilityChanged(
              ParameterGroupView groupView) {
              populateComponent(mainPanel, mode);
            }
          });
      }

      return views;
    }

    /**
     * Obtm uma viso de parmetro especfica.
     *
     * @param parameterName O nome do parmetro (No aceita {@code null}).
     *
     * @return A viso de parmetro ou {@code null} se ela no existir.
     */
    private SimpleParameterView<?> getParameterView(String parameterName) {
      for (ParameterGroupView groupView : groupViews) {
        SimpleParameterView<?> parameterView =
          groupView.getParameterView(parameterName);
        if (parameterView != null) {
          return parameterView;
        }
      }
      return null;
    }
  }

  /**
   * Painel representando a viso dos parmetros de um configurador simples em
   * modo de configurao.
   */
  private final class SimpleParametersConfigurationPanel extends
    AbstractSimpleParametersPanel {

    /**
     * Construtor
     */
    protected SimpleParametersConfigurationPanel() {
      super(Mode.CONFIGURATION);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void executeBeforeInitialization() {
      getConfigurator().resetValues();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void executeAfterInitialization() {
      getConfigurator().updateTriggers();
    }
  }

  /**
   * Painel representando a viso dos parmetros de um configurador simples em
   * modo de relatrio.
   */
  private final class SimpleParametersReportPanel extends
    AbstractSimpleParametersPanel {

    /**
     * Construtor
     */
    protected SimpleParametersReportPanel() {
      super(Mode.REPORT);
    }
  }
}
