package csbase.client.applicationmanager;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Carregador de classes especial para as bibliotecas carregadas dinamicamente
 * no cliente. Alm da lista urls para as bibliotecas,  possvel definir uma
 * lista de padres de nomenclatura de classes que podem ser carregadas do
 * classloader pai (whitelist) e uma lista anlga para classes que *no* podem
 * ser carregadas do pai (blacklist).
 */
public class ApplicationClassLoader extends URLClassLoader {

  /**
   * Lista com os padres de nome que determina que uma classe pode ser
   * carregada do classloader pai.
   */
  private final List<Pattern> whitelist;

  /**
   * Lista com os padres de nome que determina que uma classe *no* pode ser
   * carregada do classloader pai.
   */
  private final List<Pattern> blacklist;

  /**
   * Construtor.
   * 
   * @param urls as urls (diretrios ou jars) de onde sero carregadas as
   *        classes.
   * @param parent o {@link ClassLoader} pai.
   * @param whitelist lista com os padres de nome que determina que uma classe
   *        pode ser carregada do classloader pai.
   * @param blackList lista com os padres de nome que determina que uma classe
   *        *no* pode ser carregada do classloader pai.
   */
  public ApplicationClassLoader(URL[] urls, ClassLoader parent,
    List<Pattern> whitelist, List<Pattern> blackList) {
    super(urls, parent);
    /*
     * Sobrescrevendo o SecurityManager para poder executar as aplicaes
     * carregadas dinamicamente sem restries no ambiente WebStart.
     */
    System.setSecurityManager(null);
    this.whitelist = new ArrayList<Pattern>(whitelist);
    this.blacklist = new ArrayList<Pattern>(blackList);
  }

  /**
   * Construtor.
   * 
   * @param urls as urls (diretrios ou jars) de onde sero carregadas as
   *        classes.
   * @param parent o {@link ClassLoader} pai.
   */
  public ApplicationClassLoader(URL[] urls, ClassLoader parent) {
    this(urls, parent, null, null);
  }

  /**
   * Construtor.
   * 
   * @param urls as urls (diretrios ou jars) de onde sero carregadas as
   *        classes.
   */
  public ApplicationClassLoader(URL[] urls) {
    this(urls, null, null, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected synchronized Class<?> loadClass(String name, boolean resolve)
    throws ClassNotFoundException {
    //Verifica se a classe j foi carregada anteriormente.
    Class<?> c = findLoadedClass(name);
    if (c == null) {
      try {
        //Busca a classe nas URLs indicadas no construtor.
        c = findClass(name);
      }
      catch (ClassNotFoundException ignored) {
        //A classe no foi encontrada nas URLs indicadas no construtor. 
      }
      if (c == null) {
        if (isAllowed(name)) {
          //Procura a classe no classloader-pai. 
          c = getParent().loadClass(name);
        }
        if (c == null) {
          /*
           * Se o classloader-pai tambm no conseguir fazer a carga, a classe
           * realmente no est disponvel.
           */
          throw new ClassNotFoundException(name);
        }
      }
    }
    if (resolve) {
      resolveClass(c);
    }
    return c;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URL getResource(String name) {
    URL url = findResource(name);
    if (url == null && getParent() != null && isAllowed(name)) {
      url = getParent().getResource(name);
    }
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getResourceAsStream(String name) {
    URL url = getResource(name);
    try {
      if (url != null) {
        return url.openStream();
      }
      else {
        return null;
      }
    }
    catch (IOException e) {
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Enumeration<URL> getResources(String name) throws IOException {
    Enumeration<URL> res = findResources(name);
    if (res == null && getParent() != null && isAllowed(name)) {
      res = getParent().getResources(name);
    }
    return res;
  }

  /**
   * Verifica se o nome informado  uma referncia legal para dentro do
   * classloader pai.
   * 
   * @param name Nome a ser testado.
   * @return <code>true</code> se o nome pode ser buscado no classloader pai.
   */
  private final boolean isAllowed(final String name) {
    if (whitelist == null || whitelist.isEmpty()) {
      return !isBlackListed(name);
    }
    final String packageName = name.replaceAll("\\.", "/");
    final String resourceName = name.replaceAll("/", "\\.");
    for (Pattern pat : whitelist) {
      if (pat.matcher(packageName).matches()
        || pat.matcher(resourceName).matches()) {
        return !isBlackListed(name);
      }
    }
    return false;
  }

  /**
   * Verifica se o nome informado  de uma referncia ilegal para dentro do
   * classloader pai.
   * 
   * @param name Nome a ser testado.
   * @return <code>true</code> se o nome no pode ser buscado no classloader
   *         pai.
   */
  private final boolean isBlackListed(final String name) {
    if (blacklist == null || blacklist.isEmpty()) {
      return false;
    }
    final String packageName = name.replaceAll("\\.", "/");
    final String resourceName = name.replaceAll("/", "\\.");
    for (Pattern pat : blacklist) {
      if (pat.matcher(packageName).matches()
        || pat.matcher(resourceName).matches()) {
        return true;
      }
    }
    return false;
  }

}