package csbase.client.applicationmanager.resourcehelpers;

import java.io.File;
import java.io.InputStream;
import java.net.URL;

import tecgraf.javautils.version.VersionNumber;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.Application;
import csbase.client.externalresources.ExternalResources;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Utilitrio para tratamento de DTDs.
 * 
 * @author Tecgraf/PUC-Rio
 * @param <T> classe da aplicao.
 */
public class ApplicationResourceHelper<T extends Application> {

  /**
   * Registro da aplicao.
   */
  final private T application;

  /**
   * Tipo de resource envolvido.
   */
  final private ApplicationResourceType resourceType;

  /**
   * Retorna: registro de aplicao; conforme atributo {@link #application}.
   * 
   * @return o valor
   */
  protected final T getApplication() {
    return application;
  }

  /**
   * Retorna o registro de aplicao; conforme
   * {@link ApplicationManager#getApplicationRegistry(String)}.
   * 
   * @return o valor
   */
  protected final ApplicationRegistry getApplicationRegistry() {
    final ApplicationManager manager = ApplicationManager.getInstance();
    final String appId = application.getId();
    final ApplicationRegistry registry = manager.getApplicationRegistry(appId);
    return registry;
  }

  /**
   * Busca stream para o resource.
   * 
   * @param tag tag de busca
   * @return stream
   */
  final public InputStream getApplicationResourceAsStream(final String tag) {
    final String fileName = getUnversionedFileName(tag);
    final String dirName = getResourceDirectoryName();
    final String path = dirName + "/" + fileName;
    return application.getApplicationResourceAsStream(path);
  }

  /**
   * Busca stream para o resource.
   * 
   * @param tag tag de busca
   * @param version verso.
   * @return stream
   */
  final public InputStream getApplicationResourceAsStream(final String tag,
    final VersionNumber version) {
    version.check();
    final String fileName = getVersionedFileName(tag, version);
    final String dirName = getResourceDirectoryName();
    final String path = dirName + "/" + tag + "/" + fileName;
    final InputStream stream = application.getApplicationResourceAsStream(path);
    return stream;
  }

  /**
   * Busca URL para o resource.
   * 
   * @param tag tag de busca
   * @return URL
   */
  final public URL getApplicationResourceAsURL(final String tag) {
    final String fileName = getUnversionedFileName(tag);
    final String dirName = getResourceDirectoryName();
    final String path = dirName + "/" + fileName;
    return application.getApplicationResourceAsURL(path);
  }

  /**
   * Busca URL para o resource.
   * 
   * @param tag tag de busca
   * @param version verso.
   * @return URL
   */
  final public String getApplicationResourceAsURLText(final String tag,
    final VersionNumber version) {
    final String fileName = getVersionedFileName(tag, version);
    final String urlText = getURLText(fileName);
    return urlText;
  }

  /**
   * Cria um texto para montar uma URL de um arquivo (nome) com base no tipo de
   * resource, tag e estado (desenvolvimento e no Tomcat).
   * 
   * @param fileName nome do arquivo a ser buscado
   * @return texto com uma URL
   */
  final public String getURLText(final String fileName) {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = getApplicationRegistry();
    final String dirName = getResourceDirectoryName();
    final ExternalResources externalResources = ExternalResources.getInstance();
    final String sufixPath = "resources/" + dirName + "/" + fileName;
    if (externalResources.isEnabled()) {
      String codeBaseText;
      try {
        final URL codeBase = externalResources.getCodeBase();
        codeBaseText = codeBase.toString();
      }
      catch (Exception e) {
        codeBaseText = "http://";
      }
      final String urlText = codeBaseText + "/" + sufixPath;
      return urlText;
    }

    final File pwdFile = new File(".");
    final String pwdPath = pwdFile.getAbsolutePath();
    final String pckName = appManager.getApplicationFullPackageName(reg);
    final String dirPath = pckName.replaceAll("\\.", "/");
    final String filePath = pwdPath + "/" + dirPath + "/" + sufixPath;
    final String urlText = "file://" + filePath;
    return urlText;
  }

  /**
   * Cria um texto para montar uma URL de um arquivo (nome) com base no tipo de
   * resource, tag e estado (desenvolvimento e no Tomcat).
   * 
   * @return texto com uma URL
   */
  final public String getResourceURLPrefix() {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = getApplicationRegistry();
    final String dirName = getResourceDirectoryName();
    final ExternalResources externalResources = ExternalResources.getInstance();
    final String sufixPath = "resources/" + dirName;
    if (externalResources.isEnabled()) {
      String codeBaseText;
      try {
        final URL codeBase = externalResources.getCodeBase();
        codeBaseText = codeBase.toString();
      }
      catch (Exception e) {
        codeBaseText = "http://";
      }
      final String urlPrefix = codeBaseText + "/" + sufixPath + "/";
      return urlPrefix;
    }

    final File pwdFile = new File(".");
    final String pwdPath = pwdFile.getAbsolutePath();
    final String pckName = appManager.getApplicationFullPackageName(reg);
    final String dirPath = pckName.replaceAll("\\.", "/");
    final String filePath = pwdPath + "/" + dirPath + "/" + sufixPath;
    final String urlPrefix = "file://" + filePath + "/";
    return urlPrefix;
  }

  /**
   * Busca URL para o resource.
   * 
   * @param tag tag de busca
   * @param version verso.
   * @return url
   */
  final public URL getApplicationResourceAsURL(final String tag,
    final VersionNumber version) {
    version.check();
    final String fileName = getVersionedFileName(tag, version);
    final String dirName = getResourceDirectoryName();
    final String path = dirName + "/" + tag + "/" + fileName;
    final URL url = application.getApplicationResourceAsURL(path);
    return url;
  }

  /**
   * Monta nome do arquivo.
   * 
   * @param tag prefixo do arquivo.
   * @param version verso.
   * @return nome do arquivo.
   */
  final public String getVersionedFileName(final String tag,
    final VersionNumber version) {
    final String realTag = checkTag(tag);
    if (version == null) {
      final String err = "Null version detected!";
      throw new IllegalArgumentException(err);
    }
    final String vTxt = version.toString();
    final String ext = getFileExtension();
    final String fileName = realTag + "-" + vTxt + "." + ext;
    return fileName;
  }

  /**
   * Monta nome do arquivo.
   * 
   * @param tag prefixo do arquivo.
   * @return nome do arquivo.
   */
  final public String getUnversionedFileName(final String tag) {
    final String realTag = checkTag(tag);
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = getApplicationRegistry();
    final String pckText = appManager.getApplicationPackageName(reg);
    final String ext = getFileExtension();
    final String fileName = pckText + "-" + realTag + "." + ext;
    return fileName;
  }

  /**
   * Monta tag real.
   * 
   * @param tag tag de entrada
   * @return tag real (ou default se for usado {@code null}).
   */
  private String checkTag(final String tag) {
    if (tag == null) {
      final String err = "Null tag not allowed.";
      throw new IllegalArgumentException(err);
    }
    return tag.trim();
  }

  /**
   * @param tag tag
   * @return verso corrente da aplicao.
   */
  final public VersionNumber getResourceVersionFromProperty(final String tag) {
    String newTag = checkTag(tag);
    final String prefix = "resource." + getResourceTypeId();
    final String sufix = "version";
    final String appPropName = prefix + "." + newTag + "." + sufix;

    final ApplicationRegistry reg = getApplicationRegistry();
    final String versionText = reg.getStringSpecificProperty(appPropName);
    final VersionNumber version = VersionNumber.fromString(versionText);
    return version;
  }

  /**
   * Consulta o nome do diretrio do recurso (dentro de 'resources' da
   * aplicao) referente ao helper que <u>pode ser redefinido se indicado</u>
   * (por default usamos o resultado de {@link #getResourceTypeId()}).
   * 
   * 
   * @return o nome
   */
  protected String getResourceDirectoryName() {
    final String resourceTypeId = getResourceTypeId();
    return resourceTypeId;
  }

  /**
   * Consulta o id para formao da propriedade da aplicao referente ao
   * helper.
   * 
   * @return o tag
   */
  final protected String getResourceTypeId() {
    final String name = resourceType.name();
    final String lowerCaseName = name.toLowerCase();
    return lowerCaseName;
  }

  /**
   * Consulta o texto de extenso do arquivo (sem o '.' - ex.: "txt"); que
   * <u>pode ser redefinido se indicado</u> (por default usamos o resultado de
   * {@link #getResourceTypeId()}).
   * 
   * @return extenso
   */
  protected String getFileExtension() {
    final String resourceTypeId = getResourceTypeId();
    return resourceTypeId;
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   * @param resourceType tipo do recurso.
   */
  public ApplicationResourceHelper(final T application,
    ApplicationResourceType resourceType) {
    this.application = application;
    this.resourceType = resourceType;
  }

  /**
   * Retorna: tipo de resource; conforme atributo {@link #resourceType}.
   * 
   * @return o valor
   */
  public final ApplicationResourceType getResourceType() {
    return resourceType;
  }
}