/*
 * $Id$
 */
package csbase.client.applications.algorithmsmanager.actions;

/**
 * 
 * 
 * @author Tecgraf
 */
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.algorithmsmanager.dialogs.AlgorithmVersionInfoPanel;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.externalresources.ZipLocalFile;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.SingletonFileChooser;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.Utilities;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.util.FileSystemUtils;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao que exporta um arquivo de configurao.
 */
public class VersionExportAction extends CommonVersionAction {
  /** Modo de seleo para o browser de arquivos. */
  private int selectionMode;

  /**
   * Constri a ao de importar um pacote de verso de algoritmo.
   * 
   * @param versionInfoPanel painel pai que criou a ao
   * @param icon imagem da ao
   */
  public VersionExportAction(AlgorithmVersionInfoPanel versionInfoPanel,
    ImageIcon icon) {
    super(versionInfoPanel, icon);
    setSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleVersionOperation() {
    if (getSelectedAlgorithm() == null) {
      return;
    }

    if (FileSystemUtils.canRead()) {
      fileSystemExport();
    }
    else {
      appletExport(); // #TODO A remover quando todos sistemas CSBase forem
      // assinados
    }
  }

  /**
   * Atribui o modo de seleo do browser para escolha de arquivos.
   * 
   * @param selectionMode modo de seleo do browser de arquivos
   */
  public void setSelectionMode(int selectionMode) {
    this.selectionMode = selectionMode;
  }

  /**
   * Obtm o modo de seleo do browser para escolha de arquivos.
   * 
   * @return o modo de seleo do browser de arquivos
   */
  public int getSelectionMode() {
    return selectionMode;
  }

  /**
   * Cria instncias de {@link LocalFile} representando os arquivos no sistema
   * do cliente.
   * 
   * @param files O arquivos do cliente.
   * 
   * @return instncias de {@link LocalFile} representando os arquivos no
   *         sistema do cliente.
   */
  private LocalFile[] createLocalFiles(File[] files) {
    try {
      // Se tiver algum diretrio, envia tudo como um arquivo zipado.
      for (File file : files) {
        if (file.isDirectory()) {
          try {
            String fileName =
              String.format("%s.zip", file.getParentFile().getName());
            return new LocalFile[] { new ZipLocalFile(fileName, files) };
          }
          catch (OutOfMemoryError e) {
            String fatalErrorMsg =
              LNG.get("algomanager.error.upload_fatal.outOfMemory");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (IOException e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal.io");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (Exception e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
        }
      }

      List<LocalFile> sourceFiles = new ArrayList<>();
      for (File selFile : files) {
        if (selFile == null) {
          continue;
        }
        StandaloneLocalFile standAloneFile = new StandaloneLocalFile(selFile);
        sourceFiles.add(standAloneFile);
      }

      return sourceFiles.toArray(new LocalFile[sourceFiles.size()]);
    }
    catch (AccessControlException ex1) {
      // Acesso ao sistema de arquivos negado (rodando em "sandbox"): obtm
      // arquivos com a API JNLP. No  possvel a importao de diretrios.
      String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
      if (!ExternalResources.getInstance().isEnabled()) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg);
        return null;
      }
      try {
        return ExternalResources.getInstance().openMultiFileDialog(".", null);
      }
      catch (CSBaseException ex2) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg, ex2);
        return null;
      }
      catch (IOException ex3) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          LNG.get("algomanager.error.upload_io"), ex3);
        return null;
      }
    }
  }

  /**
   * Modela uma requisio de transferncia.
   */
  static class RequestTransfer {
    /**
     * Arquivo local.
     */
    LocalFile file;

    /**
     * Informao para construo do canal remoto.
     */
    RemoteFileChannelInfo info;

    /**
     * Cria uma requisio de transferncia.
     * 
     * @param file O arquivo local.
     * @param info A informao para construo do canal remoto.
     */
    RequestTransfer(LocalFile file, RemoteFileChannelInfo info) {
      this.file = file;
      this.info = info;
    }

    /**
     * Efetua a transferncia do arquivo local atravs do canal remoto.
     * 
     * @throws Exception Em caso de falha na operao.
     */
    void transfer() throws Exception {
      SyncRemoteFileChannel channel =
        new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(),
          info.getHost(), info.getPort(), info.getKey());
      channel.open(!info.isWritable());
      channel.syncTransferFrom(file.getInputStream(), 0, file.getLength());
    }
  }

  protected void appletExport() {
    AlgorithmVersionInfo selectedVersion = getSelectedVersion();
    String versionPackName = getVersionPackFileName(selectedVersion);
    String filePath =
      FileUtils.joinPath('/', selectedVersion.getVersionsDirName(),
        selectedVersion.getDirectory(),
        selectedVersion.getConfiguratorDirName(), versionPackName);
    String dialogTitle = LNG.get("algomanager.title.config.management");
    export(selectedVersion.getInfo().getId(), filePath, dialogTitle);
  }

  /**
   * Cria o arquivo a ser exportado para o sistema de arquivos local do usurio.
   * Essa tarefa deve ser feita no servidor e o arquivo transferido para o
   * cliente. Cria ento uma {@link csbase.client.desktop.LocalTask} para
   * exportar o arquivo.
   */
  private void fileSystemExport() {
    final LocalFile targetFile = getTargetFile();
    if (targetFile == null) {
      return;
    }
    if (targetFile.exists()) {
      if (!confirmOverwrite(targetFile)) {
        return;
      }
    }
    String targetFileName = null;
    try {
      targetFileName = targetFile.getName();
    }
    catch (IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    final RemoteFileChannelInfo info = getRequestInfo(targetFileName);
    if (info == null) {
      return;
    }
    Task<Void> task = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        SyncRemoteFileChannel channel = new SyncRemoteFileChannel(info);
        channel.open(true);
        long size = channel.getSize();
        OutputStream target = targetFile.getOutputStream(false);
        channel.syncTransferTo(0, size, target);
        target.flush();
        target.close();
        channel.close();
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof IOException) {
          error.printStackTrace();
          StandardDialogs.showErrorDialog(getWindow(), getName(),
            LNG.get("algomanager.error.download_fatal"));
          return;
        }
        super.handleError(error);
      }
    };
    String waitMsg = LNG.get("algomanager.msg.download_wait");
    task.execute(getWindow(), getName(), waitMsg, false, true);
  }

  /**
   * Exibe um dilogo para confirmar com o usurio se este deseja sobrescrever o
   * arquivo existente.
   * 
   * @param targetFile arquivo de destino.
   * 
   * @return <code>true</code> se o usurio confirmar, <code>false</code> caso
   *         contrrio.
   */
  private boolean confirmOverwrite(LocalFile targetFile) {
    String msg;
    try {
      msg =
        String.format(LNG.get("algomanager.msg.confirm.file_exists"),
          targetFile.getName());
      int answer = StandardDialogs.showYesNoDialog(getWindow(), getName(), msg);
      return (answer == JOptionPane.YES_OPTION);
    }
    catch (IOException ex) {
      msg = LNG.get("algomanager.error.download_io");
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(), msg, ex);
      return false;
    }
  }

  /**
   * Retorna o arquivo de destino local para a importao (se existir, ser
   * sobrescrito).
   * 
   * @return arquivo de destino local.
   */
  private LocalFile getTargetFile() {
    /**
     * Acesso ao sistema de arquivos permitido: obtm arquivos com o
     * JFileChooser.
     */
    try {
      JFileChooser chooser = SingletonFileChooser.getInstance();
      AlgorithmVersionInfo selectedVersion = getSelectedVersion();
      String versionPackName = getVersionPackFileName(selectedVersion);

      chooser.setFileSelectionMode(getSelectionMode());
      chooser.setMultiSelectionEnabled(false);

      File target =
        new File(chooser.getCurrentDirectory() + File.separator
          + versionPackName);
      chooser.setSelectedFile(target);

      int result = chooser.showSaveDialog(getWindow());
      if (result == JFileChooser.APPROVE_OPTION) {
        return new StandaloneLocalFile(chooser.getSelectedFile());
      }
      return null;
    }
    catch (AccessControlException ex) {
      // #TODO JWS
      return null;
    }
  }

  /**
   * Obtm o nome do arquivo de pacote de verso a ser sugerido para o usurio.
   * 
   * @param selectedVersion verso selecionada
   * @return o nome do arquivo de pacote
   */
  private String getVersionPackFileName(AlgorithmVersionInfo selectedVersion) {
    return selectedVersion.getInfo().getId() + "_" + selectedVersion.getId()
      + ".pva";
  }

  /**
   * Retorna um {@link RemoteFileChannelInfo}, isto , um objeto contendo as
   * informaes necessrias para solicitar a transferncia.
   * 
   * @param targetZipFileName o arquivo destino de zip.
   * 
   * @return objeto contendo as informaes necessrias para a transferncia.
   */
  private RemoteFileChannelInfo getRequestInfo(final String targetZipFileName) {
    Task<RemoteFileChannelInfo> task = new RemoteTask<RemoteFileChannelInfo>() {
      @Override
      protected void performTask() throws Exception {
        setResult(prepareDownload(targetZipFileName)); //source
      }
    };

    String waitMsg = LNG.get("algomanager.msg.upload_wait");
    if (!task.execute(getWindow(), getName(), waitMsg, false, true)) {
      return null;
    }
    return task.getResult();
  }

  protected RemoteFileChannelInfo prepareDownload(final String targetZipFileName)
    throws RemoteException {
    AlgorithmVersionInfo selectedVersion = getSelectedVersion();
    String versionPackName = getVersionPackFileName(selectedVersion);
    return ClientRemoteLocator.algorithmService.prepareDownloadVersionPackFile(
      selectedVersion.getInfo().getId(), selectedVersion.getId(),
      targetZipFileName);
  }

  /**
   * Carrega o arquivo atualmente selecionado em um navegador, permitindo sua
   * visualizao e salvamento no disco rgido. Obs: Funcionalidade disponvel
   * somente via applet.
   * 
   * @param algoId .
   * @param filePath .
   * @param dialogTitle .
   */
  protected void export(Object algoId, String filePath, String dialogTitle) {
    if (ExternalResources.getInstance().isEnabled()) {
      String[] filePathArray = Utilities.splitProjectPath(filePath);
      String urlStr =
        AlgorithmManagementProxy.retrieveDownloadURL(algoId, filePathArray,
          getWindow());
      if (urlStr == null) {
        return;
      }
      try {
        URL url = new URL(urlStr);
        ExternalResources.getInstance().showDocument(url);
      }
      catch (Exception ex) {
        StandardErrorDialogs.showErrorDialog(getWindow(), dialogTitle, ex);
      }
    }
    else {
      StandardDialogs.showInfoDialog(getWindow(), dialogTitle,
        LNG.get("algomanager.error.download_not_available"));
    }
  }

}
