package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.text.MessageFormat;
import java.util.List;
import java.util.Vector;

import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.AlgorithmsManagerAdapter;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmsManagerAction;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Classe abstrata que define um painel com abas contendo as informaes do
 * dado. Ele deve ser usado para dividir as informaes de um dado em abas, onde
 * cada uma controla suas funcionalidades.
 * 
 * Esse painel tem no mnimo uma aba contendo as informaes bsicas do dado.
 * 
 */
public abstract class CommonEditTabbedPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {

  /** Ao que originou a criao desse painel */
  private AlgorithmsManagerAction action;

  /** Painel com as abas com as informaes do dado */
  private JTabbedPane editTabbedPanel;

  /**
   * Constri o painel para edio do dado, seja uma criao de um novo dado
   * seja a alterao do dado corrente.
   * 
   * @param action ao que originou a criao desse painel
   */
  public CommonEditTabbedPanel(AlgorithmsManagerAction action) {
    super(action.getApplication());
    this.action = action;
    buildPanel();
    buildAdditionalInfoPanel();
    editTabbedPanel.setSelectedIndex(0);
    setInfoPanelTitle(getSelectedInfoPanel(), getSelectedInfoPanel().getTitle());
    action.getApplication().addAlgorithmsManagerListener(
      new AlgorithmsManagerAdapter() {

        @Override
        public void categoryUpdated(CategorySet modifiedCategorySet) {
          handleCategoryUpdated(modifiedCategorySet);
        }

        @Override
        public void categoryRemoved(Category category) {
          handleCategoryRemoved(category);
        }

        @Override
        public void categoryCreated(Category category) {
          handleCategoryCreated(category);
        }

        @Override
        public void algorithmUpdated(AlgorithmInfo algoInfo) {
          handleAlgorithmUpdated(algoInfo);
        }

        @Override
        public void algorithmRemoved(AlgorithmInfo algoInfo) {
          handleAlgorithmRemoved(algoInfo);
        }

        @Override
        public void algorithmCreated(AlgorithmInfo algoInfo) {
          handleAlgorithmCreated(algoInfo);
        }
      });
  }

  /**
   * Realiza uma ao quando uma categoria  criada na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param category categoria criada
   */
  protected void handleCategoryCreated(Category category) {
  }

  /**
   * Realiza uma ao quando uma categoria  removida na aplicao Gerenciador
   * de Algoritmos.
   * 
   * @param category categoria removida
   */
  protected void handleCategoryRemoved(Category category) {
  }

  /**
   * Realiza uma ao quando um conjunto de categorias  alterado na aplicao
   * Gerenciador de Algoritmos.
   * 
   * @param modifiedCategorySet conjunto de categorias modificadas
   */
  protected void handleCategoryUpdated(CategorySet modifiedCategorySet) {
  }

  /**
   * Realiza uma ao quando um algoritmo  alterado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo alterado
   */
  protected void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
  }

  /**
   * Realiza uma ao quando um algoritmo  criado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo criado
   */
  protected void handleAlgorithmCreated(AlgorithmInfo algoInfo) {
  }

  /**
   * Realiza uma ao quando um algoritmo  removido na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo removido
   */
  protected void handleAlgorithmRemoved(AlgorithmInfo algoInfo) {
  }

  /**
   * Obtm a ao que criou o painel.
   * 
   * @return a ao que criou o painel
   * 
   */
  public AlgorithmsManagerAction getAction() {
    return action;
  }

  /**
   * Constri o painel principal com as informaes especficas da operao de
   * edio a ser realizada sobre o dado.
   * 
   * @return o painel principal criado
   * 
   */
  protected abstract JPanel buildBasicInfoPanel();

  /**
   * Constri os painis com informaes adicionais do dado. Deve usar o mtodo
   * <code>addPanel</code> para inserir painis.
   * 
   * @see CommonEditTabbedPanel#addInfoPanel(String, CommonInfoEditPanel)
   */
  protected abstract void buildAdditionalInfoPanel();

  /**
   * Inicializa todas os painis com as informaes do dado, e realiza todas as
   * outras aes necessrias quando uma outra seleo de dado  feita.
   */
  public abstract void initializeData();

  /**
   * Confirma se pode mudar o dado selecionado. Verifica se houve alteraes em
   * algum dos painis que contm as informaes do algoritmo que est sendo
   * editado.
   * 
   * @return retorna true se confirmou a mudana para o novo dado, caso
   *         contrrio, retorna false
   */
  public abstract boolean confirmDataChanged();

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.AbstractSimpleApplicationPanel#buildPanel()
   */
  @Override
  protected void buildPanel() {
    editTabbedPanel = new JTabbedPane();
    editTabbedPanel.addTab(
      getApplication().getString("CommonEditTabbedPanel.tab.info.basic"),
      buildBasicInfoPanel());
    setLayout(new GridBagLayout());
    add(editTabbedPanel, new GBC(0, 0).both().insets(5, 5, 5, 5));
  }

  /**
   * Adiciona um painel em uma nova aba.
   * 
   * @param title ttulo da aba
   * @param panel painel com informaes de edio a ser inserido na aba
   */
  public void addInfoPanel(String title, CommonInfoEditPanel panel) {
    editTabbedPanel.addTab(title, panel);
    revalidate();
  }

  /**
   * Obtm o painel correspondente  aba selecionada.
   * 
   * @return o painel selecionado
   */
  public CommonInfoEditPanel getSelectedInfoPanel() {
    return (CommonInfoEditPanel) editTabbedPanel.getSelectedComponent();
  }

  /**
   * Estabelece um painel para ser selecionado.
   * 
   * @param infoPanel painel de informaes a ser selecionado
   * 
   */
  public void setSelectedInfoPanel(CommonInfoEditPanel infoPanel) {
    editTabbedPanel.setSelectedComponent(infoPanel);
  }

  /**
   * Atribui um novo ttulo  aba do painel especificado.
   * 
   * @param panel painel procurado
   * @param title ttulo a ser exibido
   */
  public void setInfoPanelTitle(CommonInfoEditPanel panel, String title) {
    int selectedIndex = getTabComponentIndex(panel);
    if (selectedIndex != -1) {
      editTabbedPanel.setTitleAt(selectedIndex, title);
    }
  }

  /**
   * Obtm o ndice da aba correspondente ao painel especificado.
   * 
   * @param panel painel procurado
   * 
   * @return o ndice da aba que contm o painel
   */
  private int getTabComponentIndex(CommonInfoEditPanel panel) {
    for (int i = 0; i < editTabbedPanel.getTabCount(); i++) {
      Component tabPanel = editTabbedPanel.getComponentAt(i);
      if (tabPanel != null && tabPanel.equals(panel)) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Constri um painel vazio. Pode ser necessrio para o caso de querer ocupar
   * a rea restante do painel principal.
   * 
   * @return o painel vazio
   * 
   */
  protected JPanel buildEmptyPanel() {
    return new JPanel();
  }

  /**
   * Obtm uma lista com todos os painis de informao adicionados no painel de
   * edio do dado.
   * 
   * @return uma lista com todos os painis de informao presentes nesse painel
   */
  public List<CommonInfoEditPanel> getInfoPanelList() {
    List<CommonInfoEditPanel> panelList = new Vector<>();
    Component[] components = editTabbedPanel.getComponents();
    for (Component component : components) {
      if (component instanceof CommonInfoEditPanel) {
        panelList.add((CommonInfoEditPanel) component);
      }
    }
    return panelList;
  }

  /**
   * Obtm a descrio dos painis que foram modificados. Os ttulos de cada
   * painel so exibidos em um bullet.
   * 
   * @return a descrio dos painis que foram modificados
   */
  protected String getChangedPanelsDescription() {
    String desc = "<html> <ul> ";
    // // type=\"circle\">" ou square
    List<CommonInfoEditPanel> infoPanelList = getInfoPanelList();
    for (CommonInfoEditPanel infoPanel : infoPanelList) {
      if (infoPanel.wasModified()) {
        desc += "<li>" + infoPanel.getTitle();
      }
    }
    desc += "</ul> </html>";
    return desc;
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  protected boolean confirmCancelling() {
    String changedPanelsDescription = getChangedPanelsDescription();
    String confirmMsg =
      MessageFormat.format(
        getApplication().getString("CommonEditTabbedPanel.msg.cancel.confirm"),
        new Object[] { changedPanelsDescription });
    int confirm =
      getApplication().showOptionDialog(
        confirmMsg,
        new String[] {
            getApplication().getString(
              "CommonEditTabbedPanel.msg.cancel.confirm.yes"),
            getApplication().getString(
              "CommonEditTabbedPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      // Edio foi cancelada
      List<CommonInfoEditPanel> infoPanelList = getInfoPanelList();
      for (CommonInfoEditPanel infoPanel : infoPanelList) {
        if (infoPanel.wasModified()) {
          infoPanel.setDataChanged();
        }
      }
      return true;
    }
    return false;
  }

}
