/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.table.column;

import java.util.Locale;

import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import csbase.logic.CommandInfo;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Representa uma coluna em uma tabela de comandos.<br>
 * Deve existir uma propriedade <b>column.&lt;id da coluna&gt;.name</b>, nos
 * arquivos de fraseologia da aplicao, referenciando o nome desta coluna.
 * 
 * @author Tecgraf / PUC-Rio
 */
public abstract class AbstractCommandsTableColumn implements
  IColumn<CommandInfo> {

  /**
   * Chave da propriedade que representa um valor vazio para uma coluna.
   */
  private static final String KEY_DEFAULT_EMPTY_VALUE =
    "column.default.value.empty";
  /**
   * Chave da propriedade que representa um erro como valor.
   */
  private static final String KEY_DEFAULT_ERROR_VALUE =
    "column.default.value.error";

  /**
   * Template de chave para propriedades de uma coluna.
   */
  private static final String KEY_TEMPLATE = "column.%s.%s";
  /**
   * A ser utilizada em conjunto com {@link #KEY_TEMPLATE} para se obter o nome
   * de uma coluna.
   */
  private static final String KEY_NAME = "name";

  /** Classe da propriedade do comando que  representada por esta coluna. */
  private Class<?> clazz;
  /** Identificador da coluna. */
  private String columnId;
  /**
   * Registro da aplicao que  utilizado para se obter o nome da coluna de
   * acordo com o {@link Locale}.
   */
  private ApplicationRegistry appRegistry;
  /** Flag tilizada par aindicar se a coluna  editvel. */
  private boolean editable;

  /**
   * Construtor para uma coluna no editvel.
   * 
   * @param clazz classe da propriedade do comando que  representada por esta
   *        coluna.
   * @param columnId identificador da coluna. Deve existir uma propriedade
   *        <b>column.&lt;columnId&gt;.name</b>, nos arquivos de fraseologia da
   *        aplicao, referenciando o nome desta coluna.
   * @param appRegistry registro da aplicao que ser utilizado para se obter o
   *        nome da coluna de acordo com o {@link Locale}.
   */
  protected AbstractCommandsTableColumn(Class<?> clazz, String columnId,
    ApplicationRegistry appRegistry) {

    this(clazz, columnId, appRegistry, false);
  }

  /**
   * Construtor.
   * 
   * @param clazz classe da propriedade do comando que  representada por esta
   *        coluna.
   * @param columnId identificador da coluna. Deve existir uma propriedade
   *        <b>column.&lt;columnId&gt;.name</b>, nos arquivos de fraseologia da
   *        aplicao, referenciando o nome desta coluna.
   * @param appRegistry registro da aplicao quetask.command.update.message
   *        ser utilizado para se obter o nome da coluna de acordo com o
   *        {@link Locale}.
   * @param editable <tt>true</tt> se a colundefaulta for editvel.
   */
  protected AbstractCommandsTableColumn(Class<?> clazz, String columnId,
    ApplicationRegistry appRegistry, boolean editable) {

    this.clazz = clazz;
    this.columnId = columnId;
    this.appRegistry = appRegistry;
    this.editable = editable;
  }

  /**
   * Obtm o identificador desta coluna.
   * 
   * @return o identificador desta coluna.
   */
  public final String getId() {
    return columnId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Class<?> getColumnClass() {
    return clazz;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getColumnName() {
    return getString(KEY_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean isColumnEditable() {
    return editable;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellRenderer createTableCellRenderer() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellEditor createTableCellEditor() {
    return null;
  }

  /**
   * Altera o valor da propriedade do comando, representada por esta coluna.
   * 
   * @param cmd comando representado por uma linha da tabela, ao qual se deseja
   *        obter atribuir valor a propriedade representada por esta coluna.
   * 
   * @param value o novo valor da propriedade do comando, representada por esta
   *        coluna.
   */
  @Override
  public void setValue(CommandInfo cmd, Object value) {
    /*
     * O padro  no fazer nada para facilitar a criao de colunas que no
     * sejam editveis.
     */
  }

  /**
   * Obtm o valor da propriedade do comando, representada por esta coluna.
   * 
   * @param cmd comando representado por uma linha da tabela, do qual se deseja
   *        obter a propriedade representada por esta coluna.
   * 
   * @return o valor da propriedade do comando, representada por esta coluna.
   */
  @Override
  public abstract Object getValue(CommandInfo cmd);

  /**
   * Obtm um textos especficos da coluna, de acordo com seu identificador e o
   * {@link Locale}.<br>
   * A chave  gerada fazendo
   * {@code String.format(KEY_TEMPLATE, columnId, keySufix)} para gerar a chave.
   * 
   * @param keySufix sufixo chave do texto a ser obtido de acordo com o
   *        {@link Locale}.
   * @param args argumentos a serem substitudos no texto atravs do mtodo
   *        {@link String#format(String, Object...)}.
   * 
   * @return um determinado texto de acordo como {@link Locale}.
   */
  protected final String getString(String keySufix, Object... args) {
    String key = String.format(KEY_TEMPLATE, columnId, keySufix);
    String text = this.appRegistry.getString(key);
    return String.format(text, args);
  }

  /**
   * Obtm um textos de acordo com o {@link Locale}.
   * 
   * @param key chave do texto a ser obtido de acordo com o {@link Locale}.
   * @param args argumentos a serem substitudos no texto atravs do mtodo
   *        {@link String#format(String, Object...)}.
   * 
   * @return um determinado texto de acordo como {@link Locale}.
   */
  protected final String getGenericString(String key, Object... args) {
    String text = this.appRegistry.getString(key);
    return String.format(text, args);
  }

  /**
   * Obtm o valor padro que deve ser utilizado para quando o valor 
   * {@code null}.
   * 
   * @return o valor padro que deve ser utilizado para quando o valor 
   *         {@code null}.
   */
  protected final String getDefaultEmptyValue() {
    return this.appRegistry.getString(KEY_DEFAULT_EMPTY_VALUE);
  }

  /**
   * Obtm o valor padro que deve ser utilizado para quando o valor 
   * {@code null}.
   * 
   * @return o valor padro que deve ser utilizado para quando o valor 
   *         {@code null}.
   */
  protected final String getDefaultErrorValue() {
    return this.appRegistry.getString(KEY_DEFAULT_ERROR_VALUE);
  }
}
