/**
 * $Id$
 */

package csbase.client.applications.commandsmonitor.table.script;

import java.util.Calendar;
import java.util.Date;

/**
 * Implementa uma data no modificvel para ser utilizada no valor do elemento
 * &lt;expression&gt; que ser interpretado como um script para filtrar
 * comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class ScriptDate extends Date {

  /**
   * Construtor.
   * 
   * @param date data a ser representada.
   */
  public ScriptDate(Date date) {
    super(date.getTime());
  }

  /**
   * Verifica se esta data veio depois de uma certa data.
   * 
   * @param value valor representando uma data no passado. O tipo - dia, ms,
   *        ano - desse valor ser definido posteriormente.
   * 
   * @return uma instncia de {@link Before} com a qual se pode especificar o
   *         tipo - dia, ms, ou ano - do valor.
   */
  public Before before(int value) {
    return new Before(this, value);
  }

  /**
   * Verifica se esta data veio antes de uma certa data.
   * 
   * @param value valor representando uma data no futuro. O tipo - dia, ms, ano
   *        - desse valor ser definido posteriormente.
   * 
   * @return uma instncia de {@link After} com a qual se pode especificar o
   *         tipo - dia, ms, ou ano - do valor.
   */
  public After after(int value) {
    return new After(this, value);
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setDate(int date) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setHours(int hours) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setMinutes(int minutes) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setMonth(int month) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setSeconds(int seconds) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setTime(long time) {
    throw new UnsupportedOperationException();
  }

  /**
   * @throws UnsupportedOperationException sempre que o mtodo  chamado (deprecated)
   */
  @Override
  public void setYear(int year) {
    throw new UnsupportedOperationException();
  }
}

/**
 * Implementa um comparador que verifica se uma data veio depois da outra.
 * 
 * @author Tecgraf
 */
class After extends DateComparator {
  /**
   * Construtor.
   * 
   * @param date Data principal.
   * 
   * @param value valor da data a ser comparada. O tipo ser definido
   *        posteriormente.
   */
  public After(ScriptDate date, int value) {
    super(date, value);
  }

  /**
   * @param calendar data representando o argumento {@link ScriptDate date}
   *        passado no construtor
   * @param other data representando o argumento <i>value</i> passado no
   *        construtor.
   * 
   * @return <tt>true</tt> se calendar veio depois de other.
   */
  @Override
  protected final boolean compare(Calendar calendar, Calendar other) {
    return calendar.after(other);
  }
}

/**
 * Implementa um comparador que verifica se uma data veio antes da outra.
 * 
 * @author Tecgraf
 */
class Before extends DateComparator {
  /**
   * Construtor.
   * 
   * @param date Data principal.
   * 
   * @param value valor da data a ser comparada. O tipo ser definido
   *        posteriormente.
   */
  public Before(ScriptDate date, int value) {
    super(date, value);
  }

  /**
   * @param calendar data representando o argumento {@link ScriptDate date}
   *        passado no construtor
   * @param other data representando o argumento <i>value</i> passado no
   *        construtor.
   * 
   * @return <tt>true</tt> se calendar veio antes de other.
   */
  @Override
  protected final boolean compare(Calendar calendar, Calendar other) {
    return calendar.before(other);
  }
}

/**
 * Representa um comparador de datas.
 * 
 * @author Tecgraf
 */
abstract class DateComparator {
  /**
   * Calendrio com o valor da data principal.
   */
  private Calendar calendar;
  /**
   * Inteiro representando a data a ser comparada. Ela ser formada pela data
   * corrente menos este valor, podendo ele representar dias, meses ou anos.
   */
  private int value;

  /**
   * Construtor.
   * 
   * @param date Data principal.
   * 
   * @param value valor da data a ser comparada. O tipo ser definido
   *        posteriormente.
   */
  public DateComparator(ScriptDate date, int value) {
    calendar = Calendar.getInstance();
    calendar.setTime(date);
    this.value = value;
  }

  /**
   * @return o valor da comparao de {@link ScriptDate date} com a data de
   *         <i>value</i> dias atrs.
   * @see #compare(Calendar, Calendar)
   */
  public boolean daysAgo() {
    return dateAgo(Calendar.DAY_OF_MONTH);
  }

  /**
   * @return o valor da comparao de {@link ScriptDate date} com a data de
   *         <i>value</i> meses atrs.
   * @see #compare(Calendar, Calendar)
   */
  public boolean monthsAgo() {
    return dateAgo(Calendar.MONTH);
  }

  /**
   * @return o valor da comparao de {@link ScriptDate date} com a data de
   *         <i>value</i> anos atrs.
   * @see #compare(Calendar, Calendar)
   */
  public boolean yearsAgo() {
    return dateAgo(Calendar.YEAR);
  }

  /**
   * @param field Pode ser {@link Calendar#DAY_OF_MONTH}, {@link Calendar#MONTH}
   *        , ou {@link Calendar#YEAR}.
   * @return o valor da comparao de {@link ScriptDate date} com a data de
   *         <i>value</i> <i>field</i> atrs.
   * @see #compare(Calendar, Calendar)
   */
  private boolean dateAgo(int field) {
    Calendar other = Calendar.getInstance();
    other.add(field, -value);
    return compare(calendar, other);
  }

  /**
   * Compara duas datas.<br>
   * A ser implementado pela classe que estender {@link DateComparator}.
   * 
   * @param calendar data.
   * @param other data.
   * 
   * @return <tt>true</tt> ou <tt>false</tt> dependendo da implementao.
   */
  protected abstract boolean compare(Calendar calendar, Calendar other);
}