/*
 * $Id$
 */
package csbase.client.applications.desktoplauncher;

import java.rmi.RemoteException;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.openbus.OpenBusAccessPoint;
import csbase.client.openbus.OpenBusApplicationInstanceData;
import csbase.client.openbus.OpenBusEventHandler;
import csbase.client.openbus.OpenBusStandardEvents;
import csbase.exception.CSBaseException;

/**
 * Tratdor de eventos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DesktopLauncherEventHandler extends OpenBusEventHandler {

  /**
   * Indicativo de desktop pronto
   */
  private boolean newDesktopReady = false;

  /**
   * Indicativo de aplicao inicial pronta
   */
  private boolean newApplicationReady = false;

  /**
   * Indicativo de projeto inicial aberto
   */
  private boolean newProjectReady = false;

  /**
   * Indicativo se desktop-filho deve ser fechado em conjunto com encerramento
   * da aplicao.
   */
  private boolean closeChildDesktop = true;

  /**
   * Aplicao.
   */
  final private DesktopLauncher application;

  /**
   * Envia mensagem de visibilidde do desktop
   * 
   * @param flag indicativo
   * @return indicativo de sucesso do envio da mensagem.
   */
  public final boolean fireDesktopVisibilityEvent(final boolean flag) {
    if (!isNewDesktopReady()) {
      return false;
    }
    final String sonId = getSonId();
    final OpenBusStandardEvents sendEvent =
      OpenBusStandardEvents.DESKTOP_VISIBILITY_REQ;
    final boolean sent = sendEvent.sendEvent(this, sonId, flag);
    return sent;
  }

  /**
   * Envia evento para disparo de aplicao.
   * 
   * @param appId id da aplicao.
   * @return indicativo de mensagem enviada.
   */
  final public boolean fireRemoteApplication(final String appId) {
    if (!isNewDesktopReady()) {
      return false;
    }

    final boolean appDep = application.isInitalApplicationProjectDependent();
    final String appDepTag = "application.needs.project.message";
    final String appDepMsg = getString(appDepTag) + ": " + appDep;
    application.log(appDepMsg);

    if (appDep && !isNewProjectReady()) {
      final String appDepErrTag = "application.no.project.message";
      final String appDepErrMsg = getString(appDepErrTag);
      application.log(appDepErrMsg);
      return false;
    }
    final String sonId = getSonId();
    final OpenBusStandardEvents sendType =
      OpenBusStandardEvents.APP_EXECUTION_REQ;
    final boolean sent = sendType.sendEvent(this, sonId, appId);
    return sent;
  }

  /**
   * Envia evento para disparo de aplicao.
   * 
   * @param projectId id do projeto
   * @return indicativo de mensagem enviada.
   */
  final public boolean fireRemoteProject(final String projectId) {
    if (!isNewDesktopReady()) {
      return false;
    }
    final String sonId = getSonId();
    final OpenBusStandardEvents sendType = OpenBusStandardEvents.PRJ_OPEN_REQ;
    final boolean sent = sendType.sendEvent(this, sonId, projectId);
    return sent;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #application}).
   * 
   * @return o atributo.
   */
  protected final DesktopLauncher getApplication() {
    return application;
  }

  /**
   * Obtem um texto representativo da classe do valor.
   * 
   * @param value o valor
   * @return o texto
   */
  final protected String getClassValueText(final Object value) {
    String className = "NULL";
    if (value != null) {
      final Class<? extends Object> clazz = value.getClass();
      className = clazz.getSimpleName();
    }
    return className;
  }

  /**
   * Busca (na aplicao) o id do cliente a ser lanado.
   * 
   * @return o id
   */
  final protected String getSonId() {
    final String sonId = application.getSonId();
    return sonId;
  }

  /**
   * Monta um texto de internacionalizao.
   * 
   * @param tag a tag.
   * @return o texto
   */
  final protected String getString(final String tag) {
    final String className = this.getClass().getSimpleName();
    final String formattedTag = className + "." + tag;
    if (!application.hasString(formattedTag)) {
      return LNG.get(formattedTag);
    }
    final String text = application.getString(formattedTag);
    return text;
  }

  /**
   * Obtem um texto representativo do valor.
   * 
   * @param value valor
   * @return texto
   */
  final protected String getValueText(final Object value) {
    String valueText = "null";
    if (value != null) {
      valueText = value.toString();
    }
    return valueText;
  }

  /**
   * Faz a ao inicial (depois da aplicao lanada).
   */
  final public void init() {
    application.log(getString("init.message"));
    application.log(null);
    application.log(null);
    final OpenBusStandardEvents launchType =
      OpenBusStandardEvents.DSKLAUNCHER_START_INFO;
    final String sonId = getSonId();
    launchType.sendEvent(this, sonId, null);
  }

  /**
   * Indica que a fonte do evento no  do desktop lanado.
   * 
   * @param sourceId id da fonte
   * @return indicativo
   */
  final protected boolean isEventFromMySonId(final String sourceId) {
    if (sourceId == null) {
      return false;
    }
    final String sonId = getSonId();
    return sonId.equals(sourceId);
  }

  /**
   * Consulta o atributo (ver documentao de {@link #newApplicationReady}).
   * 
   * @return o atributo.
   */
  public synchronized final boolean isNewApplicationReady() {
    return newApplicationReady;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #newProjectReady}).
   * 
   * @return o atributo.
   */
  public synchronized final boolean isNewProjectReady() {
    return newProjectReady;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #newDesktopReady}).
   * 
   * @return o atributo.
   */
  final public synchronized boolean isNewDesktopReady() {
    return newDesktopReady;
  }

  /**
   * log de descate por no ser evento direto.
   * 
   * @param sourceId origem
   * @param destId destino
   * @param type tipo do evento
   * @param indicator indicativo textual
   */
  private void logDiscard(final String sourceId, final String destId,
    final String type, final String indicator) {
    final String tag = "event.discard.message";
    final String prefix = getString(tag);
    final String headerText = prefix + ": " + type + " - " + indicator;
    final String tailText = sourceId + " --> " + destId;
    application.log(headerText + " ::: " + tailText);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logReceivedEvent(final boolean treated, final String sourceId,
    final String destId, final String type, final Object value) {
    final String className = getClassValueText(value);
    final String valueText = getValueText(value);

    final DesktopLauncherEventDirection direction =
      DesktopLauncherEventDirection.INPUT;
    final DesktopLauncherEvent launcherEvent =
      new DesktopLauncherEvent(treated, direction, sourceId, type, className,
        valueText);
    application.addLauncherEvent(launcherEvent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logSentEvent(final boolean treated, final String sourceId,
    final String destId, final String type, final Object value) {
    final String className = getClassValueText(value);
    final String valueText = getValueText(value);

    final DesktopLauncherEventDirection direction =
      DesktopLauncherEventDirection.OUTPUT;
    final DesktopLauncherEvent launcherEvent =
      new DesktopLauncherEvent(treated, direction, destId, type, className,
        valueText);
    application.addLauncherEvent(launcherEvent);
  }

  /**
   * {@inheritDoc}
   * 
   * Por default, s trata eventos diretos
   */
  @Override
  public boolean receiveEventWithBooleanValue(final String sourceId,
    final String destId, final String type, final boolean booleanValue) {
    if (!isEventFromIdentifiedSourceAndToMe(sourceId, destId)) {
      final String indicator = "(boolean)";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    if (!isEventFromMySonId(sourceId)) {
      final String indicator = "(boolean - not my son: " + sourceId + ")";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    final OpenBusStandardEvents dskVisType =
      OpenBusStandardEvents.DESKTOP_VISIBILITY_INFO;
    if (dskVisType.isMyType(type)) {
      return handleDesktopVisibilityInfo();
    }

    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithNoValue(final String sourceId,
    final String destId, final String type) {
    if (!isEventFromIdentifiedSourceAndToMe(sourceId, destId)) {
      final String indicator = "(void)";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    if (!isEventFromMySonId(sourceId)) {
      final String indicator = "(void - not my son: " + sourceId + ")";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    final OpenBusStandardEvents dskStartType =
      OpenBusStandardEvents.DESKTOP_START_INFO;
    if (dskStartType.isMyType(type)) {
      return handleDesktopStartInfo(sourceId);
    }

    final OpenBusStandardEvents dskEndType =
      OpenBusStandardEvents.DESKTOP_END_INFO;
    if (dskEndType.isMyType(type)) {
      return handleDesktopEndInfo();
    }

    return false;
  }

  /**
   * Trtador de: {@link OpenBusStandardEvents#DESKTOP_VISIBILITY_INFO}
   * 
   * @return indicativo de tratamento.
   */
  private boolean handleDesktopVisibilityInfo() {
    setCloseChildDesktopOnMyEnd(false);
    return true;
  }

  /**
   * Trtador de: {@link OpenBusStandardEvents#DESKTOP_START_INFO}
   * 
   * @param sourceId id do desktop de origem.
   * 
   * @return indicativo de tratamento.
   */
  private boolean handleDesktopStartInfo(final String sourceId) {
    if (isNewDesktopReady()) {
      final String err = getString("second.desktop.detected.message");
      application.log(err + " - " + sourceId);
      fireDesktopShutdownEvent(sourceId);
      return true;
    }

    setNewDesktopReady(true);
    final String err = getString("first.desktop.detected.message");
    application.log(err + " - " + sourceId);

    final String projId = application.getInitialProjectId();
    if (projId != null) {
      final boolean firedPrj = fireRemoteProject(projId);
      if (!firedPrj) {
        application.log(getString("project.not.fired.message"));
      }
      return true;
    }
    fireInitialApplication();
    return true;
  }

  /**
   * Dispara a aplicao inicial.
   */
  private void fireInitialApplication() {
    final String appId = application.getInitialApplicationId();
    if (appId != null) {
      final boolean firedApp = fireRemoteApplication(appId);
      if (!firedApp) {
        application.log(getString("application.not.fired.message"));
      }
    }
  }

  /**
   * Mtodo de espera para depurao de sincronismo.
   * 
   * @param sec segundos a serem esperados.
   */
  private void waitSeconds(final int sec) {
    application.log(getString("sleep.inited.message") + " (" + sec + "s)");
    try {
      Thread.sleep(1000 * sec);
    }
    catch (InterruptedException e) {
      application.log(getString("sleep.interrupted.message"));
    }
    application.log(getString("sleep.ended.message"));
  }

  /**
   * Tratador de: {@link OpenBusStandardEvents#DESKTOP_END_INFO}
   * 
   * @return indicativo
   */
  private boolean handleDesktopEndInfo() {
    setNewApplicationReady(false);
    setNewProjectReady(false);
    setNewDesktopReady(false);
    application.closeApplication();
    return true;
  }

  /**
   * Indica de desktop filho deve ser encerrado com fechamento da aplicao.
   * 
   * @return indicativo
   */
  private synchronized boolean isToCloseChildDesktopOnMyEnd() {
    return this.closeChildDesktop;
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #closeChildDesktop}).
   * 
   * @param flag o novo valor a ser ajustado.
   */
  private final synchronized void setCloseChildDesktopOnMyEnd(boolean flag) {
    this.closeChildDesktop = flag;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithStringValue(final String sourceId,
    final String destId, final String type, final String stringValue) {
    if (!isEventFromIdentifiedSourceAndToMe(sourceId, destId)) {
      final String indicator = "(string)";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    if (!isEventFromMySonId(sourceId)) {
      final String indicator = "(string - not my son: " + sourceId + ")";
      logDiscard(sourceId, destId, type, indicator);
      return false;
    }

    final OpenBusStandardEvents appStartType =
      OpenBusStandardEvents.APP_EXECUTION_SUCCESS_RSP;
    if (appStartType.isMyType(type)) {
      return handleAppExecutionSuccessResponse(stringValue);
    }

    final OpenBusStandardEvents projOpenSuccessRsp =
      OpenBusStandardEvents.PRJ_OPEN_SUCCESS_RSP;
    if (projOpenSuccessRsp.isMyType(type)) {
      return handlePrjOpenSuccessResponse(stringValue);
    }

    final OpenBusStandardEvents projOpenFailedRsp =
      OpenBusStandardEvents.PRJ_OPEN_FAILED_RSP;
    if (projOpenFailedRsp.isMyType(type)) {
      return handlePrjOpenFailedResponse(stringValue);
    }

    return false;
  }

  /**
   * Mtodo de tratamento de {@link OpenBusStandardEvents#PRJ_OPEN_SUCCESS_RSP}
   * 
   * @param projId texto recebido
   * @return indicativo de tratamento da mensagem
   */
  private boolean handlePrjOpenFailedResponse(String projId) {
    final String initProjId = application.getInitialProjectId();
    if (initProjId != null) {
      if (initProjId.equals(projId)) {
        setNewProjectReady(false);
        final String sufix = " - (" + projId + ")";
        final String tag = "init.project.failure.opened.message";
        application.log(getString(tag) + sufix);
      }
    }
    else {
      final String sufix = " - (" + projId + ")";
      final String tag = "project.failure.opened.message";
      application.log(getString(tag) + sufix);
    }
    return true;
  }

  /**
   * Mtodo de tratamento de {@link OpenBusStandardEvents#PRJ_OPEN_SUCCESS_RSP}
   * 
   * @param launchedProjId texto recebido
   * @return indicativo de tratamento da mensagem
   */
  private boolean handlePrjOpenSuccessResponse(String launchedProjId) {
    final String initProjId = application.getInitialProjectId();
    if (initProjId != null) {
      if (initProjId.equals(launchedProjId)) {
        setNewProjectReady(true);
        final String sufix = " - (" + launchedProjId + ")";
        final String tag = "init.project.correctly.opened.message";
        application.log(getString(tag) + sufix);

        fireInitialApplication();
      }
    }
    else {
      final String sufix = " - (" + launchedProjId + ")";
      final String tag = "project.correctly.opened.message";
      application.log(getString(tag) + sufix);
    }
    return true;
  }

  /**
   * Mtodo de tratamento de
   * {@link OpenBusStandardEvents#APP_EXECUTION_SUCCESS_RSP}
   * 
   * @param stringValue texto recebido
   * @return indicativo de tratamento da mensagem
   */
  private boolean handleAppExecutionSuccessResponse(final String stringValue) {
    final String[] idsFromEventValue =
      OpenBusApplicationInstanceData.getIdsFromEventValue(stringValue);
    final String badAppIdTag =
      "application.id.started.inconsistent.event.message";
    if (idsFromEventValue == null) {
      application.log(getString(badAppIdTag));
      return false;
    }
    final String lanchedAppId = idsFromEventValue[0];
    if (lanchedAppId == null) {
      application.log(getString(badAppIdTag));
      return false;
    }
    final String lanchedInstanceId = idsFromEventValue[1];
    if (lanchedInstanceId == null) {
      final String badInstIdTag =
        "instance.id.started.inconsistent.event.message";
      application.log(getString(badInstIdTag));
      return false;
    }

    final String initAppId = application.getInitialApplicationId();
    if (initAppId != null) {
      if (initAppId.equals(lanchedAppId)) {
        setNewApplicationReady(true);
        final String sufix = " - (" + lanchedInstanceId + ")";
        final String tag = "init.application.correctly.started.message";
        application.log(getString(tag) + sufix);
      }
    }
    else {
      final String sufix = " - (" + lanchedInstanceId + ")";
      final String tag = "application.correctly.started.message";
      application.log(getString(tag) + sufix);
    }
    return true;
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #newApplicationReady}).
   * 
   * @param flag o novo valor a ser ajustado.
   */
  private synchronized void setNewApplicationReady(final boolean flag) {
    this.newApplicationReady = flag;
    if (flag) {
      application.log(getString("init.application.ready.message"));
    }
    else {
      application.log(getString("init.application.not.ready.message"));
    }
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #newProjectReady}).
   * 
   * @param flag o novo valor a ser ajustado.
   */
  private synchronized void setNewProjectReady(final boolean flag) {
    this.newProjectReady = flag;
    if (flag) {
      application.log(getString("init.project.ready.message"));
    }
    else {
      application.log(getString("init.project.not.ready.message"));
    }
  }

  /**
   * Ajusta o atributo (ver documentao de {@link #newDesktopReady}).
   * 
   * @param flag o novo valor a ser ajustado.
   */
  final private synchronized void setNewDesktopReady(final boolean flag) {
    this.newDesktopReady = flag;
    if (flag) {
      application.log(getString("desktop.ready.message"));
    }
    else {
      application.log(getString("desktop.not.ready.message"));
    }
  }

  /**
   * Faz a ao final (aplicao sendo fechada).
   * 
   * @throws Exception em caso de erro de comunicao.
   */
  public void shutdown() throws Exception {
    application.log(getString("shutdown.message"));

    if (isNewDesktopReady()) {
      fireLauncherEndEvent();
    }
    else {
      final String tag = "kill.launcher.no.event.message";
      final String msg = getString(tag);
      application.log(msg);
    }

    waitSeconds(1);
    if (isToCloseChildDesktopOnMyEnd()) {
      final String sonId = getSonId();
      fireDesktopShutdownEvent(sonId);
    }

    unlinkToOpenBus();
  }

  /**
   * Envia evento de encerramento do launcher.
   */
  final protected void fireLauncherEndEvent() {
    final String sonId = getSonId();
    application.log(getString("kill.launcher.event.message"));
    final OpenBusStandardEvents launchType =
      OpenBusStandardEvents.DSKLAUNCHER_END_INFO;
    final boolean sent = launchType.sendEvent(this, sonId, null);
    if (!sent) {
      final String tag = "dsklauncher.end.event.not.sent.message";
      final String msg = getString(tag);
      application.log(msg);
    }
    else {
      final String tag = "dsklauncher.end.event.sent.message";
      final String msg = getString(tag);
      application.log(msg);
    }
  }

  /**
   * Desliga conexo com OpenBus.
   * 
   * @throws CSBaseException em caso de exceo com OpenBus
   * @throws RemoteException em caso de falha de execuo remota.
   */
  final protected void unlinkToOpenBus() throws CSBaseException,
    RemoteException {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    oap.getSession().delEventReceiver(this);
  }

  /**
   * Envia pedido de shutdown do desktop-filho.
   * 
   * @param desktopId id do desktop a ser morto.
   * 
   * @return indicativo de envio.
   */
  final public boolean fireDesktopShutdownEvent(final String desktopId) {
    final OpenBusStandardEvents evType =
      OpenBusStandardEvents.DESKTOP_SHUTDOWN_REQ;
    final boolean sent = evType.sendEvent(this, desktopId, null);
    setNewApplicationReady(false);
    setNewProjectReady(false);
    setNewDesktopReady(false);
    return sent;
  }

  /**
   * Construtor
   * 
   * @param application a aplicao.
   */
  public DesktopLauncherEventHandler(final DesktopLauncher application) {
    if (application == null) {
      final String err = "null application not allowed!";
      throw new IllegalArgumentException(err);
    }

    this.application = application;
    setNewDesktopReady(false);
    setNewApplicationReady(false);
    setNewProjectReady(false);
    application.log(getString("created.message"));
  }
}
