/*
 * $Id$
 */
package csbase.client.applications.flowapplication.commandviewer;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.rmi.RemoteException;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.view.TabType;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.client.remote.srvproxies.SchedulerProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.CommonClientProject;
import csbase.logic.User;

/**
 * Painl de Visualizao de Comandos.
 */
public abstract class CommandViewerPanel extends JPanel {
  /**
   * Localizao da barra de ferramentas.
   */
  public enum ToolBarLocation {
    /**
     * Topo
     */
    TOP,

    /**
     *  direita
     */
    RIGHT,

    /**
     *  esquerda
     */
    LEFT,

    /**
     * Abaixo
     */
    BOTTOM,
  }

  /**
   * A ao Exportar Comando.
   */
  private Action exportCommandAction;

  /**
   * A janela que chamou este dilogo.
   */
  private DesktopComponentFrame frame;

  /**
   * A ao Remover Comando.
   */
  private AbstractAction removeCommandAction;

  /**
   * A ao Matar Comando.
   */
  private AbstractAction killCommandAction;

  /**
   * A ao Exibir Comando.
   */
  private Action showCommandAction;

  /**
   * A ao Exibir Log do Comando.
   */
  private Action showCommandLogAction;

  /**
   * A tabela de visualizao de comandos.
   */
  private final CommandViewerTable table;

  /**
   * Cria o painel.
   *
   * @param frame A janela que criou esta (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   */
  protected CommandViewerPanel(DesktopComponentFrame frame, Object projectId) {
    this(frame, projectId, null, ToolBarLocation.TOP);
  }

  /**
   * Cria o painel.
   *
   * @param frame A janela (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   * @param filter O filtro (Aceita {@code null}).
   */
  protected CommandViewerPanel(DesktopComponentFrame frame, Object projectId,
    CommandPropertiesFilter filter) {
    this(frame, projectId, filter, ToolBarLocation.TOP);
  }

  /**
   * Cria o painel.
   *
   * @param frame A janela (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   * @param filter O filtro (Aceita {@code null}).
   * @param toolBarLocation indica a posio da toolbar {@link ToolBarLocation}.
   */
  protected CommandViewerPanel(DesktopComponentFrame frame, Object projectId,
    CommandPropertiesFilter filter, ToolBarLocation toolBarLocation) {
    super(new BorderLayout());
    setFrame(frame);
    createExportCommandAction();
    createRemoveCommandAction();
    createKillCommandAction();
    createShowCommandAction();
    createShowCommandLogAction();
    table = new CommandViewerTable(frame, projectId, filter);
    ListSelectionModel selectionModel = table.getSelectionModel();
    selectionModel.addListSelectionListener(createListSelectionListener());
    String location;
    boolean isHorizontal;
    switch (toolBarLocation) {
      case TOP:
        location = BorderLayout.NORTH;
        isHorizontal = true;
        break;
      case BOTTOM:
        location = BorderLayout.SOUTH;
        isHorizontal = true;
        break;

      case LEFT:
        location = BorderLayout.WEST;
        isHorizontal = false;
        break;

      case RIGHT:
        location = BorderLayout.EAST;
        isHorizontal = false;
        break;
      default:
        throw new IllegalArgumentException(
          "Posio da barra de ferramentas invlida.\n" + toolBarLocation);
    }
    add(createToolBar(isHorizontal), location);
    add(new JScrollPane(table), BorderLayout.CENTER);
  }

  /**
   * Cria o painel.
   *
   * @param frame A janela (No aceita {@code null}).
   * @param projectId O identificador do projeto (No aceita {@code null}).
   * @param toolBarLocation indica a posio da toolbar {@link ToolBarLocation}.
   */
  protected CommandViewerPanel(DesktopComponentFrame frame, String projectId,
    ToolBarLocation toolBarLocation) {
    this(frame, projectId, null, toolBarLocation);
  }

  /**
   * Obtm a ao Exportar Comando.
   *
   * @return .
   */
  public final Action getExportCommandAction() {
    return exportCommandAction;
  }

  /**
   * Obtm a ao Remover Comando.
   *
   * @return .
   */
  public final Action getRemoveCommandAction() {
    return removeCommandAction;
  }

  /**
   * Obtm a ao Matar Comando.
   *
   * @return .
   */
  public final Action getKillCommandAction() {
    return killCommandAction;
  }

  /**
   * Obtm a ao Exibir Comando.
   *
   * @return .
   */
  public final Action getShowCommandAction() {
    return showCommandAction;
  }

  /**
   * Obtm a ao Exibir Log do Comando.
   *
   * @return .
   */
  public final Action getShowCommandLogAction() {
    return showCommandLogAction;
  }

  /**
   * Inicia a exibio dos comandos.
   */
  public final void start() {
    table.start();
  }

  /**
   * Termina a exibio dos comandos.
   */
  public final void stop() {
    table.stop();
  }

  /**
   * Exporta um comando.
   *
   * @param command O comando (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} cancelamento ou
   *         falha.
   */
  protected abstract boolean exportCommand(CommandInfo command);

  /**
   * Cria o tem de menu Configurar Comando.
   */
  private void createExportCommandAction() {
    exportCommandAction =
      new AbstractAction(getMessage("label_export_action"),
        ApplicationImages.ICON_EXECUTE_CMD_16) {
        @Override
        public void actionPerformed(ActionEvent e) {
          exportSelectedCommand();
        }
      };
    exportCommandAction.putValue(Action.SHORT_DESCRIPTION,
      getMessage("tooltip_export_action"));
    exportCommandAction.setEnabled(false);
  }

  /**
   * Cria o observador de eventos de seleo na tabela de histrico de comandos.
   *
   * @return O observador.
   */
  private ListSelectionListener createListSelectionListener() {
    return new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        boolean enableSingleRowAction = 1 == table.getSelectedRows().length;
        exportCommandAction.setEnabled(enableSingleRowAction);
        showCommandAction.setEnabled(enableSingleRowAction);
        showCommandLogAction.setEnabled(enableSingleRowAction);

        List<CommandInfo> commands = table.getSelectedCommands();

        boolean enableRemove = 0 < table.getSelectedRows().length;
        for (CommandInfo command : commands) {
          enableRemove &=
            (command.getStatus() == CommandStatus.FINISHED | command
              .getStatus() == CommandStatus.SYSTEM_FAILURE);
          enableRemove &= userHasAdminRightsToCommand(command);
        }
        removeCommandAction.setEnabled(enableRemove);

        boolean enableKill = 0 < table.getSelectedRows().length;
        for (CommandInfo command : commands) {
          enableKill &=
            (command.getStatus() == CommandStatus.SCHEDULED
              | command.getStatus() == CommandStatus.UPLOADING
              | command.getStatus() == CommandStatus.INIT
              | command.getStatus() == CommandStatus.EXECUTING | command
              .getStatus() == CommandStatus.DOWNLOADING);
          enableKill &= userHasAdminRightsToCommand(command);
        }
        killCommandAction.setEnabled(enableKill);
      }

      /**
       * Retorna verdadeiro se o usurio logado no momento tem permisso para
       * cancelar/remover o comando ou falso, caso contrrio.
       *
       * @param command Comando que se deseja testar se o usurio tem permisso.
       * @return verdadeiro se o usurio tem permisso sobre o comando ou falso,
       *         caso contrrio.
       */
      private boolean userHasAdminRightsToCommand(CommandInfo command) {
        User loggedUser = User.getLoggedUser();
        CommonClientProject project = DesktopFrame.getInstance().getProject();
        return (userOwnsCommand(command) || loggedUser.isAdmin() || userOwnsProject(project));
      }
    };
  }

  /**
   * Cria o tem de menu Remover Comando.
   */
  private void createRemoveCommandAction() {
    removeCommandAction =
      new AbstractAction(getMessage("label_remove_action"),
        ApplicationImages.ICON_DELETE_16) {
        @Override
        public void actionPerformed(ActionEvent e) {
          removeSelectedCommands();
        }
      };
    removeCommandAction.setEnabled(false);
    removeCommandAction.putValue(Action.SHORT_DESCRIPTION,
      getMessage("tooltip_remove_action"));
  }

  /**
   * Cria o tem de menu Matar comando
   */
  private void createKillCommandAction() {
    killCommandAction =
      new AbstractAction(getMessage("label_kill_action"),
        ApplicationImages.ICON_STOP_16) {
        @Override
        public void actionPerformed(ActionEvent e) {
          killSelectedCommands();
        }
      };
    killCommandAction.setEnabled(false);
    killCommandAction.putValue(Action.SHORT_DESCRIPTION,
      getMessage("tooltip_kill_action"));
  }

  /**
   * Cria o tem de menu Exibir Comando.
   */
  private void createShowCommandAction() {
    showCommandAction =
      new AbstractAction(getMessage("label_show_action"),
        ApplicationImages.ICON_INFORMATION_16) {
        @Override
        public void actionPerformed(ActionEvent e) {
          showSelectedCommand(TabType.PARAMETERS);
        }
      };
    showCommandAction.setEnabled(false);
    showCommandAction.putValue(Action.SHORT_DESCRIPTION,
      getMessage("tooltip_show_action"));
  }

  /**
   * Cria o tem de menu Exibir Log de Comando.
   */
  private void createShowCommandLogAction() {
    showCommandLogAction =
      new AbstractAction(getMessage("label_show_log_action"),
        ApplicationImages.ICON_VIEWLOG_16) {
        @Override
        public void actionPerformed(ActionEvent e) {
          showSelectedCommand(TabType.LOG);
        }
      };
    showCommandLogAction.setEnabled(false);
    showCommandLogAction.putValue(Action.SHORT_DESCRIPTION,
      getMessage("tooltip_show_log_action"));
  }

  /**
   * Cria a barra de ferramentas.
   *
   * @param isHorizontal indica se a barra de ferramentas  horizontal.
   * @return A barra de ferramentas.
   */
  private JToolBar createToolBar(boolean isHorizontal) {
    JToolBar toolBar = new JToolBar();
    int orientation;
    if (isHorizontal) {
      orientation = SwingConstants.HORIZONTAL;
    }
    else {
      orientation = SwingConstants.VERTICAL;
    }
    toolBar.setOrientation(orientation);
    toolBar.add(showCommandAction);
    toolBar.add(showCommandLogAction);
    toolBar.add(exportCommandAction);
    toolBar.add(killCommandAction);
    toolBar.add(removeCommandAction);
    return toolBar;
  }

  /**
   * Se somente um comando estiver selecionado, aquele comando ser exportado.
   *
   * @return {@code false} se houver um erro, nenhum ou mais de um comando
   *         estiver selecionado.
   */
  private boolean exportSelectedCommand() {
    List<CommandInfo> commands = table.getSelectedCommands();
    if (1 == commands.size()) {
      return exportCommand(commands.get(0));
    }
    return false;
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   *
   * @param keySuffix O sufixo da chave (o nome da classe ser pr-fixado ao
   *        sufixo).
   *
   * @return A mensagem.
   */
  private String getMessage(String keySuffix) {
    return LNG.get(CommandViewerPanel.class.getSimpleName() + "." + keySuffix);
  }

  /**
   * Remove o comando selecionado.
   *
   * @return {@code false} se o usurio cancelar a operao ou se houver um
   *         erro.
   */
  private boolean removeSelectedCommands() {
    if (!warnRemoveCommand()) {
      return false;
    }
    final List<CommandInfo> commands = table.getSelectedCommands();
    if (0 == commands.size()) {
      return true;
    }

    class RemoveCommandTask extends RemoteTask<boolean[]> {
      @Override
      protected void performTask() throws RemoteException {
        setResult(CommandsCache.getInstance().removeCommands(commands));
      }
    }

    String taskTitle = getMessage("msg_removing_commands_title");
    StringBuilder sb = new StringBuilder();
    for (CommandInfo command : commands) {
      sb.append('\t').append(command.getId()).append('\n');
    }
    String taskMessage =
      String.format(getMessage("msg_removing_commands_content"), sb.toString());

    RemoveCommandTask task = new RemoveCommandTask();
    if (!task.execute(frame, taskTitle, taskMessage)) {
      return false;
    }
    boolean[] cmdsResult = task.getResult();
    // Verifica se houve erro. Se algum comando no pode ser removido.
    boolean error = false;
    for (boolean cmdResult : cmdsResult) {
      error |= !cmdResult;
    }
    /*
     * Em caso de erro, mostrar mensagem com comandos que no puderam ser
     * removidos.
     */
    if (error) {
      StringBuilder errorCmds = new StringBuilder();
      for (int inx = 0; inx < commands.size(); inx++) {
        if (!cmdsResult[inx]) {
          errorCmds.append('\t').append(commands.get(inx).getId()).append('\n');
        }
      }
      String errorMessage =
        String.format(getMessage("error_removing_commands"), errorCmds);
      showError(errorMessage);
    }

    return !error;
  }

  /**
   * Mata o comando selecionado.
   *
   * @return {@code false} se o usurio cancelar a operao ou se houver um
   *         erro.
   */
  private boolean killSelectedCommands() {
    if (!warnKillCommand()) {
      return false;
    }

    boolean success = true;

    final List<CommandInfo> commands = table.getSelectedCommands();
    boolean[] cmdsRemoveds = new boolean[commands.size()];
    for (int inx = 0; inx < commands.size(); inx++) {
      CommandInfo cmd = commands.get(inx);
      String cmdId = cmd.getId();
      if (CommandStatus.SCHEDULED.equals(cmd.getStatus())) {
        cmdsRemoveds[inx] = SchedulerProxy.removeCommand(frame, cmdId);
      }
      else {
        String sgaName = cmd.getSGAName();
        cmdsRemoveds[inx] = SGAProxy.killCommand(sgaName, cmdId);
      }
      success &= cmdsRemoveds[inx];
    }

    /*
     * Em caso de erro, mostrar mensagem com comandos que no puderam ser
     * removidos.
     */
    if (!success) {
      StringBuilder errorCmds = new StringBuilder();
      for (int inx = 0; inx < commands.size(); inx++) {
        if (!cmdsRemoveds[inx]) {
          errorCmds.append('\t').append(commands.get(inx).getId());
          String description = commands.get(inx).getDescription();
          if (description != null && description.trim().length() > 0) {
            errorCmds.append("(").append(description).append(")");
          }
          errorCmds.append('\n');
        }
      }
      String errorMessage =
        String.format(getMessage("error_killing_commands"), errorCmds);
      showError(errorMessage);
    }

    return success;
  }

  /**
   * Atribui a janela a este painl.
   *
   * @param frame A janela (No aceita {@code null}).
   */
  private void setFrame(DesktopComponentFrame frame) {
    if (frame == null) {
      throw new IllegalArgumentException("frame == null");
    }
    this.frame = frame;
  }

  /**
   * Exibe uma mensagem de erro.
   *
   * @param errorMessage A mensagem de erro (No aceita {@code null}).
   */
  private void showError(String errorMessage) {
    StandardErrorDialogs.showErrorDialog(frame, errorMessage);
  }

  /**
   * Consulta a opes do programa.
   *
   * @param msg mensagem a ser exibida.
   * @return a opo selecionada.
   */
  private int showOptionDialog(String msg) {
    final String title = frame.getTitle();
    return StandardDialogs.showYesNoDialog(frame, title, msg);
  }

  /**
   * Exibe o comando que est selecionado.
   *
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   *
   * @return {@code true} em caso de sucesso ou {@code false} em caso de erro.
   */
  private boolean showSelectedCommand(TabType preferredTab) {
    return table.showSelectedCommand(preferredTab);
  }

  /**
   * Retorna verdadeiro se o usurio logado  o dono de <b>todos</b> os comandos
   * selecionados (ou se no h comandos selecionados) ou falso, caso haja pelo
   * menos um comando dentre os selecionados que no pertence ao usurio.
   *
   * @return verdadeiro se o usurio  dono dos comandos selecionados (ou se no
   *         h comandos selecionados) ou falso caso contrrio.
   */
  protected boolean userOwnsSelectedCommands() {
    for (CommandInfo command : table.getSelectedCommands()) {
      if (!userOwnsCommand(command)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Retorna verdadeiro se o usurio logado  dono do comando ou falso, caso
   * contrrio.
   *
   * @param command O comando.
   * @return verdadeiro se o usurio  dono do comando ou falso, caso contrrio.
   */
  protected boolean userOwnsCommand(CommandInfo command) {
    User loggedUser = User.getLoggedUser();
    return loggedUser.getId().equals(command.getUserId());
  }

  /**
   * Retorna verdadeiro se o usurio logado  dono do projeto ou falso, caso
   * contrrio.
   *
   * @param project O projeto.
   * @return verdadeiro se o usurio  dono do projeto ou falso, caso contrrio.
   */
  private boolean userOwnsProject(CommonClientProject project) {
    User loggedUser = User.getLoggedUser();
    return loggedUser.getId().equals(project.getUserId());
  }

  /**
   * Avisa ao usurio que um comando est para ser removido e pede confirmao
   * da operao.
   *
   * @return {@code true} o usurio informou que quer o comando seja removido ou
   *         {@code false} o usurio informou no quer cancelar a remoo de
   *         comando.
   */
  private boolean warnRemoveCommand() {
    String message;
    if (userOwnsSelectedCommands()) {
      message = getMessage("msg_warn_command_remove");
    }
    else {
      message = getMessage("msg_warn_command_remove_not_owner");
    }
    int option = showOptionDialog(message);
    switch (option) {
      case 0:
        return true;
      case 1:
        return false;
      default:
        String errorMessage =
          String.format("Opo invlida.\nOpo: %d.\n", option);
        throw new IllegalStateException(errorMessage);
    }
  }

  /**
   * Avisa ao usurio que um comando est para ser morto e pede confirmao da
   * operao.
   *
   * @return {@code true} o usurio informou que quer o comando seja removido ou
   *         {@code false} o usurio informou no quer cancelar a remoo de
   *         comando.
   */
  private boolean warnKillCommand() {
    String message;
    if (userOwnsSelectedCommands()) {
      message = getMessage("msg_warn_command_kill");
    }
    else {
      message = getMessage("msg_warn_command_kill_not_owner");
    }
    int option = showOptionDialog(message);
    switch (option) {
      case 0:
        return true;
      case 1:
        return false;
      default:
        String errorMessage =
          String.format("Opo invlida.\nOpo: %d.\n", option);
        throw new IllegalStateException(errorMessage);
    }
  }

  /**
   * Obtm a lista de comandos selecionados no painel de visualizao.
   *
   * @return a lista de comandos selecionados.
   */
  public List<CommandInfo> getSelectedCommands() {
    return table.getSelectedCommands();
  }

}
