package csbase.client.applications.flowapplication.graph;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JToolTip;

import tecgraf.vix.Group;
import tecgraf.vix.TypeMessage;
import tecgraf.vix.TypeVO;
import csbase.client.applications.flowapplication.messages.PickElementMessage;

/**
 * <p>
 * Representa um elemento do grafo.
 * </p>
 *
 * @author lmoreira
 */
public abstract class GraphElement extends Group {
  /**
   * Cor de destaque.
   */
  protected static final Color HIGHT_LIGHT_COLOR = new Color(59, 193, 255);

  /**
   * Identificador.
   */
  private int id;

  /**
   * Flag que indica se este elemento est ou no selecionado.
   */
  private boolean isSelected;

  /**
   * Lista de observadores.
   */
  private final List<GraphElementListener> listenerList;

  /**
   * Dica.
   */
  private JToolTip toolTip;

  /**
   * Constri um elemento do grafo.
   *
   * @param graph grafo pai deste elemento.
   * @param id identificador.
   */
  protected GraphElement(final Graph graph, final int id) {
    setGraph(graph);
    this.listenerList = new LinkedList<GraphElementListener>();
    this.id = id;
  }

  /**
   * Obtm a dica que deve ser exibir para o ponto fornecido.
   *
   * @param point .
   * @return A dica ou <code>null</code> se o ponto fornecido no estiver
   * contido neste elemento ou se no h dica a ser exibida para este
   * ponto.
   */
  public abstract String getHint(Point2D point);

  /**
   * Destaca a regio deste elemento que contm o ponto passado.
   *
   * @param pt O ponto passado.
   * @return <code>true</code> sucesso ou <code>false</code> se ponto passado
   * no estiver contido neste elemento.
   */
  public abstract boolean turnOnHighlight(Point2D pt);

  /**
   * Remove o destaque deste elemento.
   */
  public abstract void turnOffHighlight();

  /**
   * Liga este elemento ao grafo.
   */
  public void attach() {
    getGraph().addElement(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    super.callbackRepaint(g);
    if (this.toolTip != null) {
      this.toolTip.setVisible(true);
    }
  }

  /**
   * Desliga este elemento do grafo.
   */
  public void deattach() {
    getGraph().removeElement(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Graphics2D getGraphics2D() {
    if (getVS() == null) {
      return null;
    }
    return getVS().getGraphics2D();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVO(final TypeMessage msg) {
    if (super.msgHandlerVO(msg)) {
      return true;
    }
    if (msg instanceof PickElementMessage) {
      return handlePickElementMessage((PickElementMessage) msg);
    }
    return false;
  }

  /**
   * Remoo de todos os listeners.
   */
  public void removeAllListeners() {
    this.listenerList.clear();
  }

  /**
   * Seleciona ou desseleciona este elemento.
   *
   * @param isSelected flag que indica se o elemento deve ser selecionado
   * <code>true</code> ou desselecionado <code>false</code>.
   */
  public void setSelected(final boolean isSelected) {
    if (isSelected() != isSelected) {
      this.isSelected = isSelected;
      notifySelecionChanged();
    }
  }

  /**
   * Adiciona um observador de elementos de grafo.
   *
   * @param listener observador de elementos de grafo.
   */
  public final void addListener(final GraphElementListener listener) {
    this.listenerList.add(listener);
  }

  /**
   * <p>
   * Arrasta este elemento ponto-a-ponto.
   * </p>
   * <p>
   * O ponto inicial  a referncia para o nicio do arraste, ao final do
   * arraste bem sucedido ele se encontrar no ponto final.
   * </p>
   * <p>
   * O drag no acompanha o grid.
   * </p>
   * <p>
   *  um template method cuja operao abstrata 
   * <code>doDrag(startPoint:Point2D, endPoint:Point2D):boolean</code>.
   * </p>
   *
   * @param startPoint O ponto inicial.
   * @param endPoint O ponto final.
   * @return <code>true</code> sucesso ou <code>false</code> se este elemento
   * no estiver selecionado, ou o x do ponto final for menor do que o x
   * do ponto limite do grafo, ou y do ponto final for menor do que o y
   * do ponto limite do grafo.
   */
  public final boolean drag(final Point2D startPoint, final Point2D endPoint) {
    if (isSelected()) {
      if (doDrag(startPoint, endPoint)) {
        for (final GraphElementListener listener : this.listenerList) {
          listener.wasDragged(this, startPoint, endPoint);
        }
        return true;
      }
    }
    return false;
  }

  /**
   * Traz elemento para frente.
   */
  public void bringToFront() {
    getGraph().bringToFront(this);
  }

  /**
   * <p>
   * Arrasta este elemento dado um deslocamento.
   * </p>
   * <p>
   * Move o elemento corrente tx  esquerda e ty acima.
   * </p>
   * <p>
   * O drag no acompanha o grid.
   * </p>
   * <p>
   *  um template method cuja operao abstrata 
   * <code>doDrag(tx:double, ty:double):void</code>.
   * </p>
   *
   * @param tx O deslocamento em x.
   * @param ty O deslocamento em y.
   * @return <code>true</code> sucesso ou <code>false</code> se este elemento
   * no estiver selecionado.
   */
  public final boolean drag(final double tx, final double ty) {
    if (isSelected()) {
      doDrag(tx, ty);
      for (final GraphElementListener listener : this.listenerList) {
        listener.wasDragged(this, tx, ty);
      }
      return true;
    }
    return false;
  }

  /**
   * <p>
   * Solta este elemento no ponto fornecido.
   * </p>
   * <p>
   * O drop  ajustado no grid.
   * </p>
   * <p>
   *  um template method cuja operao abstrata 
   * <code>doDrop(startPoint:Point2D, endPoint:Point2D):void</code>.
   * </p>
   *
   * @param pt O ponto.
   * @return <code>true</code> sucesso ou <code>false</code> se este elemento
   * no estiver selecionado.
   */
  public final boolean drop(final Point2D pt) {
    if (isSelected()) {
      doDrop(pt);
      for (final GraphElementListener listener : this.listenerList) {
        listener.wasDropped(this, pt);
      }
      return true;
    }
    return false;
  }

  /**
   * <p>
   * Solta este elemento no ponto central.
   * </p>
   * <p>
   * O drop  ajustado no grid.
   * </p>
   *
   * @return <code>true</code> sucesso ou <code>false</code> se este elemento
   * no estiver selecionado.
   */
  public final boolean drop() {
    if (isSelected()) {
      doDrop(getCenterPoint());
      for (final GraphElementListener listener : this.listenerList) {
        listener.wasDropped(this, getCenterPoint());
      }
      return true;
    }
    return false;
  }

  /**
   * Obtm o grafo.
   *
   * @return O grafo.
   */
  public final Graph getGraph() {
    return (Graph) getVS();
  }

  /**
   * Atribui o identificador do elemento, verificando se j no existe algum
   * outro elemento com o identificador selecionado.
   *
   * @param id o novo identificador.
   */
  public void setId(int id) {
    Graph graph = getGraph();
    if (graph.isIdAvailable(id)) {
      this.id = id;
    }
    else {
      throw new IllegalArgumentException("Element id already taken: " + id);
    }
  }


  /**
   * Obtm o identificador deste elemento.
   *
   * @return O identificador.
   */
  public final int getId() {
    return this.id;
  }

  //  public final int hashCode() {
  //    return this.id;
  //  }

  /**
   * Verifica se este elemento est selecionado.
   *
   * @return <code>true</code> se este elemento estiver selecionado ou
   * <code>false</code> caso contrrio.
   */
  public final boolean isSelected() {
    return this.isSelected;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final TypeVO pick(final Point2D pt) {
    TypeVO aVo = super.pick(pt);
    if (aVo == null) {
      if (contains(pt)) {
        aVo = this;
      }
    }
    return aVo;
  }

  /**
   * Remove o observador de elemento.
   *
   * @param listener O observador.
   */
  public final void removeListener(final GraphElementListener listener) {
    this.listenerList.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void repaint() {
    if (getVS() != null) {
      super.repaint();
    }
  }

  /**
   * Verifica se este elemento contm o ponto passado.
   *
   * @param pt O ponto.
   * @return <code>true</code> se este elemento contm o ponto passado ou
   * <code>false</code> caso contrrio.
   */
  protected abstract boolean contains(Point2D pt);

  /**
   * <p>
   * Arrasta este elemento de um ponto inicial a um ponto final.
   * </p>
   *
   * @param startPoint o ponto inicial.
   * @param endPoint o ponto final.
   * @return indicativo de sucesso.
   */
  protected abstract boolean doDrag(Point2D startPoint, Point2D endPoint);

  /**
   * <p>
   * Arrasta este elemento dado um deslocamento.
   * </p>
   * <p>
   * Move o elemento corrente tx  esquerda e ty acima.
   * </p>
   *
   * @param tx O deslocamento em x.
   * @param ty O deslocamento em y.
   */
  protected abstract void doDrag(double tx, double ty);

  /**
   * <p>
   * Solta este elemento no ponto fornecido.
   * </p>
   * <p>
   * Este ponto j foi previamente ajustado pelo grid.
   * </p>
   *
   * @param pt O ponto.
   */
  protected abstract void doDrop(Point2D pt);

  /**
   * Translada o elemento.
   *
   * @param pt o tamanho do translado.
   */
  protected abstract void translate(Point2D pt);

  /**
   * Reinicia este elemento.
   */
  protected void reset() {
    if (this.toolTip != null) {
      this.toolTip = null;
    }
    setSelected(false);
    turnOffHighlight();
  }

  /**
   * Obtm uma lista no modificvel de observadores de elemento.
   *
   * @return A lista de observadores.
   */
  protected final List<GraphElementListener> getListenerList() {
    return Collections.unmodifiableList(this.listenerList);
  }

  /**
   * Obtm o ponto central.
   *
   * @return O ponto central.
   */
  private Point2D getCenterPoint() {
    final Rectangle2D bounds = getBounds2D();
    final double centerX = bounds.getCenterX();
    final double centerY = bounds.getCenterY();
    return new Point2D.Double(centerX, centerY);
  }

  /**
   * Trata mensagens pegar elemento.
   *
   * @param message A mensagem.
   * @return <code>true</code> se o ponto em que a mensagem gerada est neste
   * elemento, ou <code>false</code> caso contrrio.
   */
  private boolean handlePickElementMessage(final PickElementMessage message) {
    if (contains(message.getPoint())) {
      message.setElement(this);
      return true;
    }
    return false;
  }

  /**
   * Notifica os observadores sobre mudanas na seleo deste elemento.
   */
  private void notifySelecionChanged() {
    for (final GraphElementListener listener : this.listenerList) {
      listener.wasSelected(this);
    }
  }

  /**
   * Ajuste do grafo.
   *
   * @param graph grafo
   */
  private void setGraph(final Graph graph) {
    changeVS(null, graph);
  }
}
