package csbase.client.applications.flowapplication.zoom;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * <p>
 * Modelo abstrato de zoom: Implementao padro de um modelo de zoom.
 * </p>
 * 
 * <p>
 * Esta classe  uma forma mais rpida de se implementar a interface
 * <code>ZoomModel</code>. Ela implementa vrios mtodos da interface deixando
 * apenas <code>getValue():double</code> como abstrato. Ela define mais um
 * mtodo abstrato <code>changeValue(value:double):void</code> que  uma
 * operao abstrata para o mtodo template (Template Method - GoF)
 * <code>setValue(value:double):void</code>.
 * </p>
 * 
 * @author lmoreira
 */
public abstract class AbstractZoomModel implements ZoomModel {
  /**
   * Valor mximo.
   */
  private final double maxValue;

  /**
   * Valor default.
   */
  private final double standardValue;

  /**
   * Valor mnimo.
   */
  private final double minValue;

  /**
   * Incremento/decremento de bloco.
   */
  private final double blockIncrement;

  /**
   * Incremento/decremento unitrio.
   */
  private final double unitIncrement;

  /**
   * Coleo de listeners de zoom (<code>Collection&lt;ZoomListener&gt;</code>).
   */
  private final Collection<ZoomListener> listenerCollection;

  /**
   * Constri um modelo abstrato.
   * 
   * @param minValue valor mnimo.
   * @param maxValue valor mximo.
   * @param defaultValue valor default
   * @param unitIncrement Incremento/decremento unitrio.
   * @param blockIncrement Incremento/decremento de bloco.
   * 
   * @throws IllegalArgumentException Se as relaes impostas na interface
   *         <code>ZoomModel</code> forem desrespeitadas.
   * 
   * @see csbase.client.applications.flowapplication.zoom.ZoomModel
   */
  public AbstractZoomModel(final double minValue, final double maxValue,
    final double defaultValue, final double unitIncrement,
    final double blockIncrement) {
    if (minValue <= 0.0) {
      throw new IllegalArgumentException("minValue <= 0.0");
    }
    if (maxValue <= 0.0) {
      throw new IllegalArgumentException("maxValue <= 0.0");
    }
    if (minValue > maxValue) {
      throw new IllegalArgumentException("minValue > maxValue");
    }
    if (defaultValue > maxValue) {
      throw new IllegalArgumentException("defaultValue > maxValue");
    }
    if (defaultValue < minValue) {
      throw new IllegalArgumentException("defaultValue < minValue");
    }
    if (unitIncrement <= 0.0) {
      throw new IllegalArgumentException("unitIncrement <= 0.0");
    }
    if (blockIncrement <= 0.0) {
      throw new IllegalArgumentException("blockIncrement <= 0.0");
    }
    if (unitIncrement > blockIncrement) {
      throw new IllegalArgumentException("unitIncrement > blockIncrement");
    }
    this.minValue = minValue;
    this.maxValue = maxValue;
    this.standardValue = defaultValue;
    this.blockIncrement = blockIncrement;
    this.unitIncrement = unitIncrement;
    this.listenerCollection = new LinkedList<ZoomListener>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addListener(final ZoomListener listener) {
    this.listenerCollection.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasNextBlockValue() {
    return getValue() < getMaxValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasPreviousBlockValue() {
    return getValue() > getMinValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasNextUnitValue() {
    return getValue() < getMaxValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasPreviousUnitValue() {
    return getValue() > getMinValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeListener(final ZoomListener listener) {
    this.listenerCollection.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getBlockIncrement() {
    return this.blockIncrement;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getUnitIncrement() {
    return this.unitIncrement;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getMaxValue() {
    return this.maxValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getMinValue() {
    return this.minValue;
  }

  /**
   * Retorna o valor do atributo (ver {@link #standardValue}).
   * 
   * @return o valor
   */
  @Override
  final public double getStandardValue() {
    return standardValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getNextUnitValue() {
    double nextValue = getValue() + getUnitIncrement();
    if (nextValue > this.maxValue) {
      nextValue = this.maxValue;
    }
    return nextValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getPreviousUnitValue() {
    double previousValue = getValue() - getUnitIncrement();
    if (previousValue < this.minValue) {
      previousValue = this.minValue;
    }
    return previousValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getNextBlockValue() {
    double nextValue = getValue() + getBlockIncrement();
    if (nextValue > this.maxValue) {
      nextValue = this.maxValue;
    }
    return nextValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final double getPreviousBlockValue() {
    double previousValue = getValue() - getBlockIncrement();
    if (previousValue < this.minValue) {
      previousValue = this.minValue;
    }
    return previousValue;
  }

  /**
   * <p>
   * Ajusta o valor do zoom.
   * </p>
   * 
   * <p>
   *  um Template Method (Padro GoF). Ele chama
   * <code>changeValue(value:double)</code> para efetivamente alterar o zoom,
   * porm antes de faz-lo, ele ajusta o valor do zoom para que ele seja maior
   * ou igual ao valor mnimo e menor ou igual ao valor mximo.
   * </p>
   * 
   * @param newValue o novo valor do zoom.
   */
  @Override
  public final void setValue(final double newValue) {
    double value = newValue;
    if (value > this.maxValue) {
      value = this.maxValue;
    }
    else if (value < this.minValue) {
      value = this.minValue;
    }
    changeValue(value);
    final Iterator<ZoomListener> listenerIterator =
      this.listenerCollection.iterator();
    while (listenerIterator.hasNext()) {
      final ZoomListener listener = listenerIterator.next();
      listener.wasChanged(this);
    }
  }

  /**
   * <p>
   * Modifica o valor do zoom.
   * </p>
   * 
   * <p>
   * O valor passado deve ser maior ou igual ao valor mnimo e menor ou igual ao
   * valor mximo.
   * </p>
   * 
   * <p>
   * Ele  uma operao abstrata para o mtodo
   * <code>setValue(value:double)</code>, seguindo o padro Template Method
   * (GoF).
   * </p>
   * 
   * @param value O novo valor de zoom.
   */
  protected abstract void changeValue(double value);
}
