package csbase.client.applications.flowapplication.zoom.combo;

import java.util.ArrayList;
import java.util.Collections;

import javax.swing.DefaultComboBoxModel;

import csbase.client.applications.flowapplication.zoom.ZoomListener;
import csbase.client.applications.flowapplication.zoom.ZoomModel;

/**
 * Modelo para ComboBox, que segue
 *
 * @author Tecgraf/PUC-Rio
 */
class ZoomComboModel extends DefaultComboBoxModel {

  /**
   * Valores padronizados da combo, fixos.
   */
  private ArrayList<ZoomComboItem> defaultItems =
    new ArrayList<ZoomComboItem>();

  /**
   * Valores definidos pelo usurio, via uso dos componentes de zoom, modelo
   * FIFO.
   */
  private ArrayList<ZoomComboItem> extraItems = new ArrayList<ZoomComboItem>();

  /**
   * Modelo de zoom (externo) ao qual este componente est atrelado.
   */
  private ZoomModel zoomModel;

  /**
   * Determina se as mudanas neste componente (modelo de combo) devem ser
   * repassadas para o modelo geral (externo);
   */
  private boolean notify = true;

  /**
   * Limite de itens definidos pelo usurio.
   */
  private static final int EXTRA_ITEMS_LIMIT = 3;

  /**
   * Fator de correo dos limites do modelo de zoom para a escala percetual da
   * combo.
   */
  private static int MODEL_FACTOR = 100;

  /**
   * Cria um item de Combo e atualiza as colees do modelo de combo.
   *
   * @param value Valor de zoom para o novo item.
   *
   * @return Item criado.
   */
  private ZoomComboItem createComboItem(double value) {
    ZoomComboItem newItem = new ZoomComboItem(value);
    if (getIndexOf(newItem) == -1) {
      if (extraItems.size() > EXTRA_ITEMS_LIMIT) {
        extraItems.remove(0);
      }
      extraItems.add(newItem);
      updateObjects();
    }
    return newItem;
  }

  /**
   * Atualiza a coleo interna de objetos, com a soma da coleo de valores
   * padronizados e a coleo de valores extras (definidos pelo usurio, via
   * uso).
   */
  private void updateObjects() {
    ArrayList<ZoomComboItem> arrayList =
      new ArrayList<ZoomComboItem>(defaultItems);
    arrayList.addAll(extraItems);
    Collections.sort(arrayList);
    removeAllElements();
    notify = false;
    for (ZoomComboItem item : arrayList) {
      addElement(item);
    }
    notify = true;
  }

  /**
   * Define o valor corrente da combo, por meio de um valor nmero.
   *
   * @param value Valor a ser definido.
   */
  void setValue(double value) {
    ZoomComboItem item = createComboItem(value);
    super.setSelectedItem(item);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSelectedItem(Object anObject) {
    Double value = null;
    if (anObject instanceof ZoomComboItem) {
      super.setSelectedItem(anObject);
      value = ((ZoomComboItem) anObject).getValue();
    }
    else if (anObject instanceof String) {
      ZoomComboItem item = parseInput(anObject);
      if (item != null) {
        super.setSelectedItem(item);
        value = item.getValue();
      }
    }
    if (notify && value != null) {
      zoomModel.setValue(value);
    }
  }

  /**
   * Traduz o input do usurio de texto para nmero e o define na combo. Esta
   * operao segue as seguintes regras:
   *
   * -- Para um valor inferior ao valor mnimo do modelo de zoom, retorna o
   * valor mnimo do modelo de zoom.
   *
   * -- Para um valor superior ao valor mximo do modelo de zoom, retorna o
   * valor mximo do modelo de zoom.
   *
   * -- Para um valor que no pode ser convertido em um nmero real positivo,
   * retorna nulo.
   *
   * @param anObject candidato a ser definido.
   *
   * @return Item de combo correspondente ao valor recebido.
   */
  private ZoomComboItem parseInput(Object anObject) {
    String newVal = (String) anObject;
    newVal = newVal.trim();
    char lastChar = newVal.charAt(newVal.length() - 1);
    if (lastChar == '%') {
      newVal = newVal.substring(0, newVal.length() - 1);
    }
    try {
      double value = Double.parseDouble(newVal);
      value = value / MODEL_FACTOR;
      if (value > zoomModel.getMaxValue()) {
        return createComboItem(zoomModel.getMaxValue());
      }
      if (value < zoomModel.getMinValue()) {
        return createComboItem(zoomModel.getMinValue());
      }
      return createComboItem(value);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Inicializa a coleo de itens padro.
   */
  private void createDefaultItems() {
    notify = false;
    defaultItems.add(new ZoomComboItem(0.5));
    defaultItems.add(new ZoomComboItem(0.75));
    ZoomComboItem zoomComboItem = new ZoomComboItem(1);
    defaultItems.add(zoomComboItem);
    defaultItems.add(new ZoomComboItem(1.5));
    defaultItems.add(new ZoomComboItem(2));
    defaultItems.add(new ZoomComboItem(4));

    Collections.sort(defaultItems);
    setSelectedItem(zoomComboItem);
    notify = true;
  }

  /**
   * Construtor.
   *
   * @param zoomModel Modelo de zoom que rege o comportamento deste componente.
   */
  ZoomComboModel(ZoomModel zoomModel) {
    this.zoomModel = zoomModel;
    createDefaultItems();
    for (ZoomComboItem item : defaultItems) {
      addElement(item);
    }
    zoomModel.addListener(new ZoomModelComboListner(this));
  }

  /**
   * Wrapper para as informaes de zoom exibidas na combo.
   *
   * @author Tecgraf/PUC-Rio
   */
  private class ZoomComboItem implements Comparable<ZoomComboItem> {

    /**
     * Valor real de zoom no modelo.
     */
    private double value;

    /**
     * Construtor.
     *
     * @param value Valor real de zoom no modelo.
     */
    ZoomComboItem(double value) {
      this.value = value;
    }

    /**
     * Exibe o valor na forma percentual, com o smbolo '%'.
     */
    @Override
    public String toString() {
      int a = (int) (MODEL_FACTOR * value);
      return a + "%";
    }

    /**
     * @return Retorna o valor real de zoom do modelo deste item.
     */
    private double getValue() {
      return value;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
      if (obj instanceof ZoomComboItem) {
        double other = ((ZoomComboItem) obj).getValue();
        other *= 10;
        other = Math.round(other) / 10.0;

        double myself = getValue() * 10;
        myself = Math.round(myself) / 10.0;

        if (myself == other) {
          return true;
        }
      }
      return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int compareTo(ZoomComboItem o) {
      if (getValue() < o.getValue()) {
        return -1;
      }
      if (getValue() > o.getValue()) {
        return 1;
      }
      return 0;
    }
  }

  /**
   * Ouvinte de mudanas no modelo de zoom, responsvel por notificar o modelo
   * da combo.
   *
   * @author Tecgraf/PUC-Rio
   */
  private class ZoomModelComboListner implements ZoomListener {

    /**
     * Modelo de zoom da combo.
     */
    private ZoomComboModel zoomComboModel;

    /**
     * Construtor.
     *
     * @param model Modelo de zoom da combo.
     */
    ZoomModelComboListner(ZoomComboModel model) {
      this.zoomComboModel = model;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChanged(ZoomModel model) {
      double value = model.getValue();
      zoomComboModel.setValue(value);
    }
  }
}
