/*
 * $Id$
 */
package csbase.client.applications.imageviewer.effects;

import java.awt.image.BufferedImage;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRootPane;

import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationTask;
import csbase.client.applications.imageviewer.ImageViewer;

/**
 * Descrio do efeito
 * 
 * @author Tecgraf/PUC-Rio
 */
abstract public class AbstractEffect {

  /**
   * Aplicao.
   */
  final private ImageViewer application;

  /**
   * Label de preview.
   */
  private JLabel previewLabel = null;

  /**
   * Monta o texto de internacionalizao, usando uma chave sem o prefixo padro
   * que  o nome da classe. Obtm dinamicamente o nome da classe para usar como
   * prefixo da busca.
   * 
   * @param tag o sufixo da chave. A chave ser: \"
   *        {@code getClass().getSimpleName()}.{@code tag}\"
   * @return o texto
   */
  final protected String getClassString(final String tag) {
    final String className = getClass().getSimpleName();
    final String text = application.getString(className + "." + tag);
    return text;
  }

  /**
   * Monta o texto de internacionalizao.
   * 
   * @param tag tag
   * @return o texto
   */
  final protected String getString(final String tag) {
    final String text = application.getString(tag);
    return text;
  }

  /**
   * Ajuste do label de preview.
   * 
   * @param previewLabel label
   */
  public final void setPreviewLabel(final JLabel previewLabel) {
    this.previewLabel = previewLabel;
  }

  /**
   * Aplica a transformao
   * 
   * @throws Exception em caso de erro.
   */
  public final void applyImageTransformation() throws Exception {
    final ApplicationFrame frame = application.getApplicationFrame();
    final JRootPane component = frame.getRootPane();
    final BufferedImage result = createNewImage(component);
    if (result == null) {
      return;
    }
    application.setCurrentImage(result);
  }

  /**
   * Aplica a transformao ao local de preview.
   * 
   * @return indicativo de imagem atualizada.
   */
  public final boolean updatePreview() {
    if (previewLabel == null) {
      // Garantia que campos do panel que tenham listeners de atualizao
      // no afetem o preview ainda no construdo. Aqui, no se trata
      // de ocultar o erro.
      return false;
    }

    try {
      final BufferedImage result = createNewImage(previewLabel);
      if (result != null) {
        previewLabel.setIcon(new ImageIcon(result));
        return true;
      }
      previewLabel.setIcon(ApplicationImages.ICON_QUESTION_32);
      return false;
    }
    catch (final Exception e) {
      previewLabel.setIcon(ApplicationImages.ICON_ERROR_32);
      previewLabel.setText(e.getMessage());
      return false;
    }
  }

  /**
   * Monta uma nova imagem dom base no efeito.
   * 
   * @param cmp componente pai da task.
   * @return a imagem
   * @throws Exception em caso de falha
   */
  private BufferedImage createNewImage(final JComponent cmp) throws Exception {
    final BufferedImage image = application.getCurrentBufferedImage();
    if (image == null) {
      return null;
    }

    final String msg = application.getString("effect.task.msg");
    final ApplicationTask<BufferedImage> task =
      new ApplicationTask<BufferedImage>(application, msg) {
        @Override
        protected void handleError(final Exception exception) {
          // O tratamento de exceo  feito por quem chama.
        }

        @Override
        protected void performTask() {
          final BufferedImage newImage = transformImage(image);
          setResult(newImage);
        }
      };

    final boolean executed = task.execute();
    if (!executed) {
      final Exception error = task.getError();
      if (error != null) {
        throw error;
      }

      final String err = application.getString("effect.error.msg");
      throw new Exception(err);
    }
    final BufferedImage result = task.getResult();
    return result;
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  protected AbstractEffect(final ImageViewer application) {
    this.application = application;
  }

  /**
   * Monta um painel para o efeito.
   * 
   * @return o painel
   */
  abstract public JPanel getParameterPanel();

  /**
   * Funo de transformao da imagem.
   * 
   * @param image imagem original
   * @return nova imagem
   */
  abstract protected BufferedImage transformImage(final BufferedImage image);

  /**
   * Aplica a transformao ao local de preview.
   */
  abstract public void resetParameters();

  /**
   * Retorna aplicao associada ao efeito
   * 
   * @return aplicao
   */
  protected ImageViewer getApplication() {
    return application;
  }

}
