/*
 * $Id$
 */
package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.client.applications.projectsmanager.usersearch.UserDialog;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Ao que adiciona usurios compartilhados a projetos
 *
 * @author Tecgraf/PUC-Rio
 */
public class AppendUsersAction extends ProjectsManagerAction {

  /**
   * Boolean que indica se o modo de acesso dos usurios ser "apenas leitura"
   * ou "leitura e escrita".
   */
  final private boolean readOnly;

  /**
   * Construtor
   *
   * @param projectsManager A aplicao
   * @param readOnly Boolean que indica se o modo de acesso dos usurios ser
   *        "apenas leitura" ou "leitura e escrita".
   */
  public AppendUsersAction(final ProjectsManager projectsManager,
    final boolean readOnly) {
    super(projectsManager);
    this.readOnly = readOnly;

    if (readOnly) {
      putValue(NAME, getString("AppendUsersAction.readonly.name"));
    }
    else {
      putValue(NAME, getString("AppendUsersAction.readwrite.name"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() == 0) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final String title = getString("AppendUsersAction.selection.title");
    final Set<Object> ids =
      UserDialog.showUserSelectionDialog(projectsManager.getApplicationFrame(),
        projectsManager.getUsers(), true, title);

    if (ids == null) {
      return;
    }

    runTask(projects, ids);
    projectsManager.refreshProjectsTable();
    projectsManager.refreshInfoPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ProjectsManagerUI.ADD_SHARED;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    User loggedUser = User.getLoggedUser();
    Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      ProjectSpaceAllocation spaceAllocation = pmd.getProjectSpaceAllocation();
      ProjectsManagerScope scope = pmd.getScope();

      boolean isAdmin = loggedUser.isAdmin();
      boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

      /*
       * Para que um usurio possa tornar um projeto pblico  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio seja o administrador ou o dono do projeto.
       */
      if (spaceAllocation.isOpenable() && (isAdmin || isOwner)
        && scope == ProjectsManagerScope.SHARED) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("AppendUsersAction.project.selection.denied.message")
        + getString("AppendUsersAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("AppendUsersAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("AppendUsersAction.project.selection.some.denied.message")
        + getString("AppendUsersAction.project.requirements.message");
      int res =
        StatusDialog
        .showSomeOkDialog(
          projectsManager,
          someDeniedMessage,
          prjList,
          getString("AppendUsersAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }
    return prjList;
  }

  /**
   * Roda e trata a execuo da task.
   *
   * @param projects Lista de projetos vlidos para a operao.
   * @param ids Conjunto de ids dos usurios a serem adicionados aos projetos.
   * @throws Exception em caso de erro.
   */
  private void runTask(List<ProjectsManagerData> projects, Set<Object> ids)
    throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final AppendUsersTask aut =
      new AppendUsersTask(projectsManager, projects, ids, readOnly);

    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    aut.execute(frame, projectsManager.getName(),
      getString("AppendUsersAction.message"));
    if (aut.wasCancelled()) {
      final String err = getString("AppendUsersAction.cancelled.message");
      aut.showError(err);
      return;
    }
    if (aut.getStatus() != true) {
      final Exception exception = aut.getError();
      throw exception;
    }
    updateLocalProjects(projects, ids);
  }

  /**
   * Atualiza localmente os projetos modificados.
   *
   * @param projects Os projetos modificados.
   * @param ids Os ids dos usurios que ganharam acesso ao projeto.
   */
  private void updateLocalProjects(final List<ProjectsManagerData> projects,
    final Set<Object> ids) {
    for (ProjectsManagerData pmd : projects) {
      final Set<Object> userIds;
      if (readOnly) {
        userIds = pmd.getUsersRO();
      }
      else {
        userIds = pmd.getUsersRW();
      }
      userIds.addAll(ids);
    }
  }

}

/**
 * Task que adiciona usurios a um conjunto de projetos (transformando o escopo
 * do projeto em "compartilhado" se este ainda no for).
 *
 * @author Tecgraf
 */
class AppendUsersTask extends ProjectsManagerTask<Void> {

  /**
   * Lista de projetos aos quais os usurios sero adicionados.
   */
  final private List<ProjectsManagerData> projects;

  /**
   * Lista de usurios que sero adicionados aos projetos.
   */
  final private Set<Object> userIds;

  /**
   * Modo de acesso dos usurios sobre os projetos.
   */
  final private boolean readOnly;

  /**
   * Construtor.
   *
   * @param projectsManager A aplicao.
   * @param projects Lista de projetos aos quais os usurios sero adicionados.
   * @param userIds Lista de usurios que sero adicionados aos projetos.
   * @param readOnly Modo de acesso dos usurios sobre os projetos.
   */
  public AppendUsersTask(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projects, final Set<Object> userIds,
    final boolean readOnly) {
    super(projectsManager);
    this.projects = projects;
    this.userIds = userIds;
    this.readOnly = readOnly;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    if (userIds == null) {
      return;
    }
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    if (readOnly) {
      for (ProjectsManagerData pmd : projects) {
        CommonClientProject ccp = psi.openProject(pmd.getProjectId(), false);
        ccp.setSharingType(SharingType.PARTIAL);
        Set<Object> newUserIds = new HashSet<Object>();
        newUserIds.addAll(userIds);
        Set<Object> usersRO = ccp.getUsersRO();
        if (usersRO != null) {
          newUserIds.addAll(usersRO);
          ccp.updateUsersRO(newUserIds);
        }
        ccp.close(true);
      }
    }
    else {
      for (ProjectsManagerData pmd : projects) {
        CommonClientProject ccp = psi.openProject(pmd.getProjectId(), false);
        ccp.setSharingType(SharingType.PARTIAL);
        Set<Object> newUserIds = new HashSet<Object>();
        newUserIds.addAll(userIds);
        Set<Object> usersRW = ccp.getUsersRW();
        if (usersRW != null) {
          newUserIds.addAll(ccp.getUsersRW());
          ccp.updateUsersRW(newUserIds);
        }
        ccp.close(true);
      }
    }
  }
}
