package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.ProjectStatusModificationDialog;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.StatusModificationTask;

/**
 * Ao para alterao do status de alocao de "Sem alocao" para
 * "Aguardando alocao".
 * 
 * @author jnlopes
 * 
 */
public class ChangeStatusLockAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ChangeStatusLockAction(ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    int allocatedSpace =
      ProjectStatusModificationDialog.openDialog(projectsManager, projects);
    if (allocatedSpace <= 0) {
      return;
    }

    boolean ret = runTask(projects, allocatedSpace);
    if (ret) {
      updateStatusOfLocalProjects(projects, allocatedSpace);
      projectsManager.refreshProjectsTable();
    }
    else {
      String errMsg =
        getString("ChangeStatusLockAction.error.allocation.message");
      String errTitle =
        getString("ChangeStatusLockAction.error.allocation.title");
      StandardDialogs.showErrorDialog(projectsManager.getApplicationFrame(),
        errTitle, errMsg);
    }
  }

  /**
   * Define o status de alocao dos projetos validados como
   * <code>ProjectSpaceAllocation.ALLOCATED</code> e o espao de alocao.
   * 
   * @param projects Projetos validados.
   * @param lockedSpace Espao a ser solicitado para alocao.
   */
  private void updateStatusOfLocalProjects(
    final List<ProjectsManagerData> projects, final int lockedSpace) {
    for (ProjectsManagerData pmd : projects) {
      pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.WAITING_ALLOCATION);
      pmd.setRequestedSpace(lockedSpace);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @param lockedSpace Espao a ser solicitado para alocao.
   * @return Boolean indicando se a operao foi bem sucedida.
   */
  private boolean runTask(List<ProjectsManagerData> projects, int lockedSpace) {
    ProjectsManager projectsManager = getProjectsManager();
    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    StatusModificationTask smt =
      new StatusModificationTask(projectsManager, projects,
        ProjectSpaceAllocation.WAITING_ALLOCATION, lockedSpace);
    String taskTitle = getString("ChangeStatusLockAction.running.title");
    String taskMsg = getString("ChangeStatusLockAction.running.message");
    smt.execute(appFrame, taskTitle, taskMsg);
    if (smt.wasCancelled()) {
      String err = getString("ChangeStatusLockAction.cancelled.message");
      smt.showError(err);
      return false;
    }
    if (smt.getStatus() != true) {
      Exception exception = smt.getError();
      String err =
        getString("ChangeStatusLockAction.allocation.status.error.message");
      String msg = err + "\n" + exception.getMessage();
      StandardDialogs.showErrorDialog(appFrame, msg, "");
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();

    prjList.addAll(projectList);
    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      ProjectSpaceAllocation spaceAllocation = pmd.getProjectSpaceAllocation();
      if (spaceAllocation != ProjectSpaceAllocation.NO_ALLOCATION) {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
      else {
        i++;
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("ChangeStatusLockAction.project.selection.denied.message")
          + getString("ChangeStatusLockAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("ChangeStatusLockAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("ChangeStatusLockAction.project.selection.some.denied.message")
          + getString("ChangeStatusLockAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("ChangeStatusLockAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return null;
  }

}
