/*
 * $Id:$
 */

package csbase.client.applications.projectsmanager.usersearch;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.logic.UserOutline;

/**
 * Dilogo para visualizao, seleo e atribuio de usurios.
 * 
 * @author jnlopes
 */
public class UserDialog extends JDialog {

  /**
   * Flag que indica se o usurio confirmou (true) ou cancelou (false) a edio.
   */
  private boolean userConfirmed;

  /**
   * Flag que indica se o dilogo j foi fechado. Se sim, algumas informaes j
   * esto disponveis (como p.ex. as listas de usurios).
   */
  private boolean isClosed;

  /**
   * Boolean que indica que a visualizao do dilogo exige uma resposta
   * ("confirmar" / "cancelar") do usurio.
   */
  final private boolean confirmation;

  /**
   * Usurios selecionados na tabela do dilogo.
   */
  private Set<Object> selectedUsers;

  /**
   * Painel para visualizao / seleo de usurios.
   */
  protected final UserPanel userPanel;

  /**
   * Construtor.
   * 
   * @param owner - janela-me
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @param confirmation Boolean que indica que a visualizao do dilogo exige
   *        uma resposta ("confirmar" / "cancelar") do usurio.
   */
  public UserDialog(final Window owner, final List<UserOutline> allUsers,
    final boolean showPermissions, final boolean confirmation) {
    super(owner);
    this.confirmation = confirmation;
    this.userPanel = getUserPanel(allUsers, showPermissions);
  }

  /**
   * Construtor.
   * 
   * @param owner - janela-me
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @param confirmation Boolean que indica que a visualizao do dilogo exige
   *        uma resposta ("confirmar" / "cancelar") do usurio.
   */
  public UserDialog(final Window owner, final List<UserOutline> allUsers,
    final Set<Object> usersRO, final Set<Object> usersRW,
    final boolean showPermissions, final boolean confirmation) {
    super(owner);
    this.confirmation = confirmation;
    this.userPanel = getUserPanel(allUsers, usersRO, usersRW, showPermissions);
  }

  /**
   * Construtor.
   * 
   * @param owner - janela-me
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @param confirmation Boolean que indica que a visualizao do dilogo exige
   *        uma resposta ("confirmar" / "cancelar") do usurio.
   * @param fixedAccess Indica se os usurios so todos membros ou no-membros
   *        dos projetos.
   */
  public UserDialog(final Window owner, final List<UserOutline> allUsers,
    final Set<Object> usersRO, final Set<Object> usersRW,
    final boolean showPermissions, final boolean confirmation,
    final boolean fixedAccess) {
    super(owner);
    this.confirmation = confirmation;
    this.userPanel =
      getUserPanel(allUsers, usersRO, usersRW, showPermissions, fixedAccess);
  }

  /**
   * Retorna um painel de usurios dada a lista de usurios, lista de usurios
   * com leitura e escrita, o esquema de permisses e indicao se so membro ou
   * no
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @param fixedAccess Indica se os usurios so todos membros ou no-membros
   *        dos projetos.
   * @return um painel de usurios dada a lista de usurios, lista de usurios
   *         com leitura e escrita, o esquema de permisses e indicao se so
   *         membro ou no
   */
  protected UserPanel getUserPanel(List<UserOutline> allUsers,
    Set<Object> usersRO, final Set<Object> usersRW, boolean showPermissions,
    boolean fixedAccess) {
    return new UserPanel(allUsers, usersRO, usersRW, showPermissions,
      fixedAccess);
  }

  /**
   * Retorna um painel de usurios dada a lista de usurios, lista de usurios
   * com leitura e escrita e o esquema de permisses
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @return um painel de usurios dada a lista de usurios, lista de usurios
   *         com leitura e escrita e o esquema de permisses
   */
  protected UserPanel getUserPanel(List<UserOutline> allUsers,
    Set<Object> usersRO, final Set<Object> usersRW, boolean showPermissions) {
    return new UserPanel(allUsers, usersRO, usersRW, showPermissions);
  }

  /**
   * Retorna um painel de usurios dada a lista de usurios e o esquema de
   * permisses
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @return um painel de usurios dada a lista de usurios e o esquema de
   *         permisses
   */
  protected UserPanel getUserPanel(List<UserOutline> allUsers,
    boolean showPermissions) {
    return new UserPanel(allUsers, showPermissions);
  }

  /**
   * Mtodo esttico que apresenta um dilogo com tabela de usurios.
   * 
   * @param frame Frame principal
   * @param rawUsers lista de usurios
   */
  public static void showUserListDialog(final JFrame frame,
    final List<UserOutline> rawUsers) {

    final String title = LNG.get("UserDialog.users.title");
    final UserDialog ud = new UserDialog(frame, rawUsers, false, false);
    ud.setTitle(title);
    ud.execute();
  }

  /**
   * Mtodo esttico que apresenta um dilogo para visualizao de usurios, com
   * seus respectivos modos de acesso. A tabela no  editvel.
   * 
   * @param frame Frame principal
   * @param rawUsers lista de usurios
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   */
  public static void showProjectUsersDialog(final JFrame frame,
    final List<UserOutline> rawUsers, final Set<Object> usersRO,
    final Set<Object> usersRW) {

    final UserDialog ud =
      new UserDialog(frame, rawUsers, usersRO, usersRW, true, false, true);

    ud.setTitle(LNG.get("UserDialog.project.users.title"));
    ud.userPanel.enableSharedUsersPanel(false);
    ud.execute();
  }

  /**
   * Mtodo esttico que apresenta um dilogo para visualizao e seleo de
   * usurios e retorna a lista de usurios selecionados.
   * 
   * @param owner Frame principal
   * @param rawUsers lista de usurios
   * @param multipleSelection Indica se a tabela deve permitir seleo mltipla
   *        de usurios.
   * @param title Ttulo do dilogo.
   * @return Lista de usurios selecionados.
   */
  public static Set<Object> showUserSelectionDialog(final Window owner,
    final List<UserOutline> rawUsers, final boolean multipleSelection,
    final String title) {

    final UserDialog ud = new UserDialog(owner, rawUsers, false, true);
    ud.setModalityType(ModalityType.APPLICATION_MODAL);
    ud.userPanel.setTableSelection(multipleSelection);
    ud.setTitle(title);
    ud.execute();
    return ud.getSelectedUsers();
  }

  /**
   * Retorna os usurios selecionados.
   * 
   * @return Usurios selecionados.
   */
  public Set<Object> getSelectedUsers() {
    return selectedUsers;
  }

  /**
   * Cria a interface.
   */
  protected void createGUI() {

    // interceptamos tentativas de fechar a janela
    setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        cancel();
      }
    });

    setLayout(new GridBagLayout());

    // adiciona verticalmente 1 ou mais painis para entrada de dados
    final int y = addInputPanels(0);

    // painel inferior com os botes da tela
    final JPanel buttonsPanel = createButtonsPanel();
    final GBC gbc = new GBC(0, y + 1).northwest().bottom(5).horizontal();
    add(buttonsPanel, gbc);
  }

  /**
   * Adiciona um ou mais painis para entrada de dados, verticalmente, de cima
   * para baixo, a partir de uma "linha" especfica de um
   * <code>GridBagLayout</code>.
   * 
   * @param ypos - linha inicial do layout a partir da qual devem ser inseridos
   *        os painis
   * @return y + (nmero de painis adicionados), para ser usado como referncia
   *         para insero de novos painis
   */
  private int addInputPanels(final int ypos) {
    int y = ypos;
    final GBC gbc = new GBC(0, y++).both().insets(10);
    add(userPanel, gbc);
    return y;
  }

  /**
   * Cria o painel inferior com os botes "Confirmar" e "Cancelar".
   * 
   * @return painel inferior com os botes "Confirmar" e "Cancelar"
   */
  private JPanel createButtonsPanel() {
    final JPanel panel = new JPanel();

    if (!confirmation) {
      final JButton closeButton = new JButton(LNG.get("UserDialog.button.ok"));
      closeButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          dispose();
        }
      });
      panel.add(closeButton);
      return panel;
    }

    final JButton confirmButton =
      new JButton(LNG.get("UserDialog.button.confirm"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (canConfirm()) {
          confirm();
        }
      }
    });
    final JButton cancelButton =
      new JButton(LNG.get("UserDialog.button.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        cancel();
      }
    });

    final JButton[] buttons = new JButton[] { confirmButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);

    panel.add(confirmButton);
    panel.add(cancelButton);
    return panel;
  }

  /**
   * Aes associadas ao boto "Confirmar".
   */
  protected void confirm() {
    userConfirmed = true;
    selectedUsers = userPanel.getSelectedUsers();
    userPanel.fillUserSets();
    close();
  }

  /**
   * Avalia se a tela pode ser confirmada.
   * 
   * @return true se no houver nenhuma inconsistncia e a tela pode ser
   *         confirmada
   */
  private boolean canConfirm() {
    return true;
  }

  /**
   * Aes associadas ao boto "Cancelar".
   */
  private void cancel() {
    close();
  }

  /**
   * Aes associadas ao fechamento da janela.
   */
  private void close() {
    setVisible(false);
    isClosed = true;
  }

  /**
   * Exibe a tela.
   * 
   * @return true se o usurio confirmou a edio, false caso tenha cancelado
   */
  public boolean execute() {
    createGUI();
    setModalityType(ModalityType.APPLICATION_MODAL);
    setSize(new Dimension(500, 400));
    setLocationRelativeTo(getOwner());
    setVisible(true);
    return userConfirmed;
  }

  /**
   * Obtm o conjunto de usurios com acesso RO ao projeto.
   * 
   * @return conjunto de usurios com acesso RO ao projeto
   */
  public Set<Object> getUsersRO() {
    checkIsClosed();
    return userPanel.getUsersRO();
  }

  /**
   * Garante que o dilogo j foi fechado.
   */
  private void checkIsClosed() {
    if (!isClosed) {
      throw new IllegalStateException("close() deveria ter sido chamado");
    }
  }

  /**
   * Retorna true se o dilogo foi fechado com confirmao
   * 
   * @return true se o dilogo foi fechado com confirmao
   */
  public boolean wasConfirmed() {
    return userConfirmed;
  }

  /**
   * Obtm o conjunto de usurios com acesso RW ao projeto.
   * 
   * @return conjunto de usurios com acesso RW ao projeto
   */
  public Set<Object> getUsersRW() {
    checkIsClosed();
    return userPanel.getUsersRW();
  }
}
