package csbase.client.applications.serverdiagnostic.properties;

import java.awt.GridBagLayout;
import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.logic.diagnosticservice.PropertyInfo;

/**
 *
 * Tabela que exibe as propriedades. Pode ter associado um painel de detalhes da
 * propriedade cuja linha est selecionada na tabela.
 *
 * @author Tecgraf PUC-Rio
 */
public class PropertiesTable extends SortableTable {

  /** Sem valor definido */
  private static final String NO_VALUE = "--";

  /** O painel de detalhamento */
  private JPanel detailPanel;
  /** As informaes sobre as propriedades */
  private PropertyInfo[] propertiesInfo;
  /** O label com o valor da propriedade configurada no servio */
  private JLabel serviceValueLabel = new JLabel(NO_VALUE);
  /** O label com o valor da propriedade configurada no servidor */
  private JLabel serverValueLabel = new JLabel(NO_VALUE);
  /** O label com o valor da propriedade configurada no sistema */
  private JLabel systemValueLabel = new JLabel(NO_VALUE);
  /** O label com o valor da propriedade configurada por linha de comando */
  private JLabel commandLineValueLabel = new JLabel(NO_VALUE);

  /**
   * Construtor.
   *
   * @param propertiesInfo informaes sobre as propriedades do servidor
   * @param createDetailPanel {@code true} se um painel de detalhamento deve ser
   *        criado e {@code false} caso contrrio
   */
  public PropertiesTable(PropertyInfo[] propertiesInfo,
    boolean createDetailPanel) {
    this.propertiesInfo = propertiesInfo;
    PropertiesTableModel model = new PropertiesTableModel(this.propertiesInfo);
    setModel(model);
    setRowSelectionAllowed(true);
    this.getSelectionModel().setSelectionMode(
      ListSelectionModel.SINGLE_SELECTION);
    if (createDetailPanel) {
      detailPanel = makeDetailPanel();
      this.getSelectionModel().addListSelectionListener(
        new ListSelectionListener() {
          @Override
          public void valueChanged(ListSelectionEvent event) {
            if (event.getValueIsAdjusting()) {
              return;
            }
            updateDetailPanel();
          }
        });
    }
    setComparator(0, new Comparator<String>() {
      @Override
      public int compare(String o1, String o2) {
        return compareStringsWithNumbers(o1, o2);
      }
    });
    sort();
  }

  /**
   * Comparador de Strings que leva em considerao valores numricos.
   *
   * Exemplo: A-2 ser menor que A-12.
   *
   * Nota: S funciona para inteiros, ou seja, A-12.2 ser menor que A-12.13.
   *
   * @param s1 String (contendo nmeros ou no).
   * @param s2 String (contendo nmeros ou no).
   * @return @see {@link Comparable#compareTo(Object)}
   *
   *         TODO Avaliar otimizaes (a recursividade demanda criao de novas
   *         strings).
   *
   *         TODO Avaliar necessidade de considerar nmeros de ponto flutuante.
   *
   *         TODO cdigo est duplicado, existe tambm em
   *         infogrid.client.util.gui.MultipleValuesField
   *
   */
  private int compareStringsWithNumbers(String s1, String s2) {
    Pattern pattern = Pattern.compile("\\d+");
    Matcher matcher1 = pattern.matcher(s1);
    Matcher matcher2 = pattern.matcher(s2);

    if (matcher1.find() && matcher2.find()) { // As duas Strings contm nmeros.

      int m1 = matcher1.start();
      int m2 = matcher2.start();

      if ((m1 > 0) ^ (m2 > 0)) {
        return s1.compareTo(s2); // Um prefixo numrico, outro no, comparao de strings tradicional.
      }

      if (m1 > 0 && m2 > 0) { // Nenhum prefixo  numrico.

        String str1 = s1.substring(0, m1 - 1).toString();
        String str2 = s2.substring(0, m2 - 1).toString();
        int result = str1.compareTo(str2);
        if (result != 0) {
          return result; // Prefixos diferentes.
        }
      }

      // Prefixos iguais ou nulos.

      String number1 = matcher1.group();
      String number2 = matcher2.group();
      Double d1 = Double.parseDouble(number1);
      Double d2 = Double.parseDouble(number2);
      int result = Double.compare(d1, d2);

      if (result == 0) { // Nmeros iguais, verificar o restante.

        return compareStringsWithNumbers(s1.substring(matcher1.start()
          + number1.length()), s2
          .substring(matcher2.start() + number2.length()));
      }
      return result; // Nmeros diferentes.
    }
    return s1.compareTo(s2); // Sem nmeros em (ao menos) uma das Strings.
  }

  /**
   * Contri um painel com as informaes detalhadas das propriedades
   *
   * @return o painel
   */
  private JPanel makeDetailPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(new JLabel(LNG.get("PropertiesTable.service.label")), new GBC(0,
      0).none().west().insets(5, 5, 0, 0));
    panel.add(serviceValueLabel, new GBC(1, 0).horizontal().west().insets(5, 5,
      0, 0));
    panel.add(new JLabel(LNG.get("PropertiesTable.server.label")),
      new GBC(0, 1).none().west().insets(5, 5, 0, 0));
    panel.add(serverValueLabel, new GBC(1, 1).horizontal().west().insets(5, 5,
      0, 0));
    panel.add(new JLabel(LNG.get("PropertiesTable.system.label")),
      new GBC(0, 2).none().west().insets(5, 5, 0, 0));
    panel.add(systemValueLabel, new GBC(1, 2).horizontal().west().insets(5, 5,
      0, 0));
    panel.add(new JLabel(LNG.get("PropertiesTable.command.line.label")),
      new GBC(0, 3).none().west().insets(5, 5, 5, 0));
    panel.add(commandLineValueLabel, new GBC(1, 3).horizontal().west().insets(
      5, 5, 5, 0));
    panel.setBorder(BorderFactory.createTitledBorder(LNG
      .get("PropertiesTable.detail.title")));
    return panel;
  }

  /**
   * Atualiza as informaes no painel de detalhamento das propriedades, de
   * acordo com a linha selecionada na tabela de propriedades.
   */
  private void updateDetailPanel() {
    int row = this.getSelectedRow();
    if (row < 0) {
      serviceValueLabel.setText(NO_VALUE);
      serverValueLabel.setText(NO_VALUE);
      systemValueLabel.setText(NO_VALUE);
      commandLineValueLabel.setText(NO_VALUE);
    }
    else {
      serviceValueLabel
        .setText(propertiesInfo[row].serviceValue == null ? NO_VALUE
          : propertiesInfo[row].serviceValue);
      serverValueLabel
        .setText(propertiesInfo[row].serverValue == null ? NO_VALUE
          : propertiesInfo[row].serverValue);
      systemValueLabel
        .setText(propertiesInfo[row].systemValue == null ? NO_VALUE
          : propertiesInfo[row].systemValue);
      commandLineValueLabel
        .setText(propertiesInfo[row].commandLineValue == null ? NO_VALUE
          : propertiesInfo[row].commandLineValue);
    }
  }

  /**
   * Obtm o painel que exibe o detalhamento da propriedade cuja linha est
   * selecionada na tabela.
   *
   * @return o painel de detalhes, atualizado em funo da linha selecionada na
   *         tabela.
   */
  public JPanel getPropertiesDetailPanel() {
    return this.detailPanel;
  }
}
