package csbase.client.facilities.commandtable;

import javax.swing.Icon;

import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe utilitria para criao de representaes textuais e/ou grficas dos
 * estados de um comando.
 */
public class CommandStatusHelper {

  /**
   * Cria um {@link CommandStatusCellData} com a representao do estado do
   * comando passado.
   *
   * @param commandInfo Informaes bsicas sobre o comando.
   * @return Um CommandStatusCellData com a representao do estado do comando.
   */
  public static CommandStatusCellData createCellData(CommandInfo commandInfo) {
    Icon icon = getImage(commandInfo);
    String tooltip = getDescription(commandInfo);
    return new CommandStatusCellData(commandInfo, icon, tooltip);
  }

  /**
   * Retorna a descrio textual correspondente ao estado atual do comando
   * passado.
   *
   * @param commandInfo Informaes bsicas sobre o comando.
   * @return A descrio textual do estado do comando.
   */
  public static String getDescription(CommandInfo commandInfo) {
    CommandStatus status = commandInfo.getStatus();
    if (status == null) {
      return "";
    }
    if (status == CommandStatus.FINISHED) {
      CommandFinalizationType finalizationType =
        commandInfo.getFinalizationType();
      String description =
        getDescriptionFromFinalizationType(finalizationType);
      if (commandInfo.hasWarnings()) {
        description += " " + getMessage("status_finished_warning");
      }
      return description;
    }

    if (commandInfo.isValid()) {
      if (commandInfo.isQueued()) {
        return getMessage("status_queued");
      }
      return getDescriptionFromCommandStatus(status);
    }
    return getMessage("status_disconnected");
  }

  /**
   * Retorna a descrio textual do {@link CommandStatus} de um comando.
   *
   * @param status O status do comando.
   * @return A descrio do status do comando.
   */
  public static String getDescriptionFromCommandStatus(CommandStatus status) {
    switch (status) {
      case SCHEDULED:
        return getMessage("status_scheduled");
      case INIT:
        return getMessage("status_init");
      case UPLOADING:
        return getMessage("status_uploading");
      case EXECUTING:
        return getMessage("status_executing");
      case DOWNLOADING:
        return getMessage("status_downloading");
      case SYSTEM_FAILURE:
        return getMessage("system_failure");
      case FINISHED: {
        return getMessage("status_finished");
      }
      default:
        return "";
    }
  }

  /**
   * Retorna a descrio textual do {@link CommandFinalizationType} de um
   * comando.
   *
   * @param finalizationType O tipo de finalizao do comando.
   * @return A descrio do tipo de finalizao do comando.
   */
  public static String getDescriptionFromFinalizationType(
    CommandFinalizationType finalizationType) {
    if (null == finalizationType) {
      return getMessage("status_finished");
    }
    switch (finalizationType) {
      case UNKNOWN:
        return getMessage("status_finished_unknown");
      case EXECUTION_ERROR:
        return getMessage("status_finished_error");
      case END:
        return getMessage("status_finished_end");
      case SUCCESS:
        return getMessage("status_finished_success");
      case FAILED:
        return getMessage("status_finished_failed");
      case KILLED:
        return getMessage("status_finished_killed");
      case LOST:
        return getMessage("status_finished_lost");
      case NO_EXIT_CODE:
        return getMessage("status_finished_no_code");
      default:
        return getMessage("status_finished");
    }
  }

  /**
   * Retorna a imagem correspondente ao estado atual do comando passado.
   *
   * @param commandInfo Informaes bsicas sobre o comando.
   * @return A imagem correspondente ao estado do comando.
   */
  public static Icon getImage(CommandInfo commandInfo) {
    CommandStatus status = commandInfo.getStatus();
    if (status == null) {
      return CommandImages.ICON_CMD_NONE;
    }
    if (status == CommandStatus.FINISHED) {
      CommandFinalizationType finalizationType =
        commandInfo.getFinalizationType();
      if (finalizationType == CommandFinalizationType.SUCCESS && commandInfo
        .getFinalizationInfo().hasWarnings()) {
        return CommandImages.ICON_CMD_WARNING;
      }
      return getImageFromFinalizationType(finalizationType);
    }

    if (commandInfo.isValid()) {
      if (commandInfo.isQueued()) {
        return CommandImages.ICON_CMD_QUEUED;
      }
      return getImageFromCommandStatus(status);
    }
    return CommandImages.ICON_CMD_DISCONNECTED;
  }

  /**
   * Retorna a imagem que representa o {@link CommandStatus} de um comando.
   *
   * @param status O status do comando.
   * @return A imagem que representa oo status do comando.
   */
  public static Icon getImageFromCommandStatus(CommandStatus status) {
    switch (status) {
      case DOWNLOADING:
        return CommandImages.ICON_CMD_DOWNLOADING;
      case EXECUTING:
        return CommandImages.ICON_CMD_EXECUTING_ANIMATED;
      case INIT:
        return CommandImages.ICON_CMD_INIT;
      case SCHEDULED:
        return CommandImages.ICON_CMD_SCHEDULED;
      case UPLOADING:
        return CommandImages.ICON_CMD_UPLOADING;
      case FINISHED:
        return CommandImages.ICON_CMD_END;
      case SYSTEM_FAILURE:
        return CommandImages.ICON_CMD_SYSTEM_FAILURE;
      default:
        return CommandImages.ICON_CMD_NONE;
    }
  }

  /**
   * Retorna a imagem que representa o {@link CommandFinalizationType} de um
   * comando.
   *
   * @param finalizationType O tipo de finalizao do comando.
   * @return A imagem que representa o tipo de finalizao do comando.
   */
  public static Icon getImageFromFinalizationType(
    CommandFinalizationType finalizationType) {
    if (null == finalizationType) {
      return CommandImages.ICON_CMD_END;
    }
    switch (finalizationType) {
      case END:
        return CommandImages.ICON_CMD_END;
      case EXECUTION_ERROR:
        return CommandImages.ICON_CMD_ERROR;
      case FAILED:
        return CommandImages.ICON_CMD_FAIL;
      case KILLED:
        return CommandImages.ICON_CMD_KILLED;
      case LOST:
        return CommandImages.ICON_CMD_LOST;
      case SUCCESS:
        return CommandImages.ICON_CMD_SUCCESS;
      case UNKNOWN:
        return CommandImages.ICON_CMD_UNKNOWN;
      case NO_EXIT_CODE:
        return CommandImages.ICON_CMD_NO_CODE;
      default:
        return CommandImages.ICON_CMD_NONE;
    }
  }

  /**
   * Retorna a string traduzida da mensagem que corresponde  chave passada.
   *
   * @param keySuffix Chave para a mensagem
   * @return A string traduzida da mensagem
   */
  private static String getMessage(String keySuffix) {
    return LNG.get(CommandStatusHelper.class.getSimpleName() + "." + keySuffix);
  }

}
