package csbase.client.facilities.configurabletable.UI;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.JTabbedPane;

import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Painel de abas especfico para tabelas configurveis.<br>
 * Para criar uma aba  necessrio uma lista de tabelas que sero exibidas
 * naquela aba. Alm disso,  possvel definir se as tabelas estaro alinhadas
 * horizontalmente ou verticalmente.
 * 
 * @see Panel4Tables ConfigurableTable UIFactory
 * 
 * @author Tecgraf
 */
public class TabbedPane4Tables extends JTabbedPane implements UI4Tables {

  /**
   * Identificador do painel de abas.
   */
  private String id;

  /**
   * Mapa que associa cada painel a sua respectiva aba.
   */
  private Map<String, Panel4Tables> panelByTab;

  /**
   * Construtor padro.
   * 
   * @param id - identificador do painel de abas.
   */
  public TabbedPane4Tables(String id) {
    super();

    if (id == null) {
      throw new IllegalArgumentException("id no pode ser nulo.");
    }

    this.id = id;
    this.panelByTab = new LinkedHashMap<String, Panel4Tables>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * Cria uma nova aba.
   * 
   * @param title - ttulo da aba.
   * @param orientation - orientation da aba.
   */
  public void addTab(String title, ORIENTATION orientation) {
    panelByTab.put(title, new Panel4Tables(getId(), orientation));
  }

  /**
   * Adiciona uma tabela em uma aba especfica do painel de abas.
   * 
   * @param <T> - tipo dos dados exibidos na tabela.
   * @param title - ttulo da aba.
   * @param table - tabela a ser adicionada.
   * @param label - rtulo da tabela.
   * @param dataType - tipo dos dados exibidos na tabela.
   */
  public <T> void addTable(String title, ConfigurableTable<?> table,
    String label, Class<T> dataType) {

    if (!isTabExist(title)) {
      throw new IllegalArgumentException("No existe aba com ttulo: " + title);
    }

    panelByTab.get(title).addTable(table, label, dataType);
  }

  /**
   * Verifica se existe uma aba com ttulo <code>title</code>.
   * 
   * @param title - ttulo da aba.
   * @return <code>true</code> se existir uma aba com ttulo <code>title</code>,
   *         <code>false</code> caso contrrio.
   */
  private boolean isTabExist(String title) {
    return panelByTab.containsKey(title);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigurableTable<?> getTableById(String tableId) {

    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {
      ConfigurableTable<?> table = entry.getValue().getTableById(tableId);
      if (table != null) {
        return table;
      }
    }

    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T> ConfigurableTable<T> getTableById(String tableId, Class<T> dataType) {

    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {
      Panel4Tables panel = entry.getValue();
      ConfigurableTable<T> table = panel.getTableById(tableId, dataType);
      if (table != null) {
        return table;
      }
    }

    return null;
  }

  /**
   * Retorna todas as tabelas da aba.
   * 
   * @param title - ttulo da aba.
   * 
   * @return todas as tabelas da aba.
   */
  public List<ConfigurableTable<?>> getTablesByTab(String title) {

    if (panelByTab.containsKey(title)) {
      return panelByTab.get(title).getAllTables();
    }

    return null;
  }

  /**
   * Retorna todas as tabelas da aba <code>title</code> que exibem objetos do
   * tipo <code>T</code>.
   * 
   * @param <T> - tipo dos objetos que so exibidos pelas tabelas.
   * @param title - ttulo da aba.
   * @param dataType - tipo dos objetos que so exibidos pelas tabelas.
   * @return tabelas da aba <code>title</code> que exibem objetos do tipo
   *         <code>T</code>.
   */
  public <T> List<ConfigurableTable<T>> getTablesByTab(String title,
    Class<T> dataType) {

    if (panelByTab.containsKey(title)) {
      return panelByTab.get(title).getAllTables(dataType);
    }

    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConfigurableTable<?>> getTablesFromSelectedComponent() {
    int selectedTab = getSelectedIndex();
    String tabTitle = getTitleAt(selectedTab);

    return getTablesByTab(tabTitle);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T> List<ConfigurableTable<T>> getTablesFromSelectedComponent(
    Class<T> dataType) {
    int selectedTab = getSelectedIndex();
    String tabTitle = getTitleAt(selectedTab);

    return getTablesByTab(tabTitle, dataType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConfigurableTable<?>> getAllTables() {
    List<ConfigurableTable<?>> allTables =
      new ArrayList<ConfigurableTable<?>>();

    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {
      allTables.addAll(entry.getValue().getAllTables());
    }

    return allTables;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T> List<ConfigurableTable<T>> getAllTables(Class<T> dataType) {

    List<ConfigurableTable<T>> allTables =
      new ArrayList<ConfigurableTable<T>>();

    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {
      allTables.addAll(entry.getValue().getAllTables(dataType));
    }

    return allTables;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTableLabel(String tableId) {
    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {
      String result = entry.getValue().getTableLabel(tableId);
      if (result != null) {
        return result;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void buildUI() {
    removeAll();

    for (Entry<String, Panel4Tables> entry : panelByTab.entrySet()) {

      String tabTitle = entry.getKey();
      Panel4Tables panel = entry.getValue();

      panel.buildUI();
      add(tabTitle, panel);

    }
  }
}
